#!/usr/bin/env python
#
# This file is part of vospace-transfer-service
# Copyright (C) 2021 Istituto Nazionale di Astrofisica
# SPDX-License-Identifier: GPL-3.0-or-later
#

import logging

from config import Config
from db_connector import DbConnector
from job_queue import JobQueue
from job import Job
from redis_log_handler import RedisLogHandler
from redis_rpc_server import RedisRPCServer


class GroupRwRPCServer(RedisRPCServer):

    def __init__(self, host, port, db, rpcQueue):
        self.type = "group_rw"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("scheduling")
        self.maxReadyJobs = params.getint("max_ready_jobs")
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  2,
                                  self.logger)
        self.groupRwReadyQueue = JobQueue("group_rw_ready")
        super(GroupRwRPCServer, self).__init__(host, port, db, rpcQueue)

    def callback(self, requestBody):
        # 'requestType' and 'vospacePath' attributes are mandatory
        if "requestType" not in requestBody or "vospacePath" not in requestBody:
            errorMsg = "Malformed request, missing parameters."
            self.logger.error(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 1,
                         "errorMsg": errorMsg }
        elif (requestBody["requestType"] == "GRPR_ADD" or 
              requestBody["requestType"] == "GRPR_DEL" or
              requestBody["requestType"] == "GRPW_ADD" or 
              requestBody["requestType"] == "GRPW_DEL"):
            requestType = requestBody["requestType"]
            jobType = requestType.split('_')[0]
            vospacePath = requestBody["vospacePath"]
            groupname = requestBody["groupName"]            
            try:
                nodeExists = self.dbConn.nodeExists(vospacePath)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            try:
                queueLen = self.groupRwReadyQueue.len()
            except Exception:
                errorMsg = "Cache error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 3,
                             "errorMsg": errorMsg }
                return response
            if not nodeExists:
                errorMsg = "VOSpace path not found."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 4,
                             "errorMsg": errorMsg }
            elif queueLen >= self.maxReadyJobs:
                errorMsg = "'group_rw_ready' queue is full, please, retry later."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 5,
                             "errorMsg": errorMsg }
            else:
                jobObj = Job()
                jobObj.setType("vos_group")
                jobInfo = requestBody.copy()
                jobObj.setInfo(jobInfo)
                jobObj.setPhase("QUEUED")
                try:
                    creatorId = self.dbConn.getCreatorId(vospacePath)
                except Exception:
                    errorMsg = "Database error."
                    self.logger.exception(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 2,
                                 "errorMsg": errorMsg }
                    return response
                jobObj.setOwnerId(creatorId)
                try:
                    self.dbConn.insertJob(jobObj)
                except Exception:
                    errorMsg = "Database error."
                    self.logger.exception(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 2,
                                 "errorMsg": errorMsg }
                    return response
                try:
                    self.groupRwReadyQueue.insertJob(jobObj)
                except Exception:
                    errorMsg = "Cache error."
                    self.logger.exception(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 3,
                                 "errorMsg": errorMsg }
                    return response
                response = { "responseType": f"{jobType}_STARTED" }
        elif requestBody["requestType"] == "GRPR_LST":
            vospacePath = requestBody["vospacePath"]     
            try:
                if not self.dbConn.nodeExists(vospacePath):
                    errorMsg = "VOSpace path not found."
                    self.logger.error(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 4,
                                 "errorMsg": errorMsg }
                else:
                    result = self.dbConn.getGroupRead(vospacePath)
                    response = { "responseType": "GRPR_LST_DONE",
                                 "groupList": result }
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
        elif requestBody["requestType"] == "GRPW_LST":
            vospacePath = requestBody["vospacePath"]
            try:
                if not self.dbConn.nodeExists(vospacePath):
                    errorMsg = "VOSpace path not found."
                    self.logger.error(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 4,
                                 "errorMsg": errorMsg }
                else:
                    result = self.dbConn.getGroupWrite(vospacePath)
                    response = { "responseType": "GRPW_LST_DONE",
                                 "groupList": result }
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
        else:
            errorMsg = "Unkown request type."
            self.logger.error(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 5,
                         "errorMsg": errorMsg }
        return response

    def run(self):
        self.logger.info(f"Starting RPC server of type {self.type}...")
        super(GroupRwRPCServer, self).run()
