#!/usr/bin/env python
#
#  TODO:
#  - error codes and status codes list and description
#  - check what happens if the user folder does not exist (e.g. /home/ccarbone)
#
#

import json
import logging
import os
import sys

from redis_log_handler import RedisLogHandler
from redis_rpc_server import RedisRPCServer
from config import Config
from db_connector import DbConnector
from job import Job
from job_queue import JobQueue
from system_utils import SystemUtils


class DataRPCServer(RedisRPCServer):

    def __init__(self, host, port, db, rpcQueue):
        self.type = "data"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  2)
        params = config.loadSection("transfer_node")
        self.storageStorePath = params["store_path"]
        params = config.loadSection("scheduling")
        self.maxPendingJobs = params.getint("max_pending_jobs")
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        self.pendingQueueWrite = JobQueue("write_pending")
        self.systemUtils = SystemUtils()
        super(DataRPCServer, self).__init__(host, port, db, rpcQueue)

    def callback(self, requestBody):
        # 'requestType' and 'userName' attributes are mandatory
        if "requestType" not in requestBody or "userName" not in requestBody:
            response = { "responseType":"ERROR",
                         "errorCode": 1,
                         "errorMsg": "Malformed request, missing parameters." }
        elif self.pendingQueueWrite.len() >= self.maxPendingJobs:
            response = { "responseType": "ERROR",
                         "errorCode": 2,
                         "errorMsg": "Pending queue is full, please, retry later." }
        elif requestBody["requestType"] == "CSTORE" or requestBody["requestType"] == "HSTORE":
            user = requestBody["userName"]
            userInDb = self.dbConn.userExists(user)
            if requestBody["requestType"] == "CSTORE":
                storageList = self.dbConn.getStorageListByType("cold")
            else:
                storageList = self.dbConn.getStorageListByType("hot")
            #folderPath = "/home/" + user + "/store"
            folderPath = self.storageStorePath.replace("{username}", user)
            userInfo = self.systemUtils.userInfo(user)
            # Check if the user exists on the transfer node and is registered in the database
            if not userInfo or not userInDb:
                response = { "responseType": "ERROR",
                             "errorCode": 3,
                             "errorMsg": "The user does not exist on the transfer node or is not registered in the database." }
            else:
                uid = os.stat(folderPath).st_uid
                gid = os.stat(folderPath).st_gid
                # Check if uid and gid match and avoid privilege escalation
                if uid == userInfo[1] and gid == userInfo[2] and uid != 0 and gid != 0:
                    # If write permissions are set and the 'store' folder is not empty,
                    # it means that data is ready to be copied, otherwise, nothing can
                    # be done until the write permissions are restored or new data is
                    # copied on the transfer node by the user.
                    if os.access(folderPath, os.W_OK) and os.listdir(folderPath):
                        response = { "responseType": "STORE_ACK",
                                     "storageList": storageList }
                    elif os.access(folderPath, os.W_OK) and not os.listdir(folderPath):
                        response = { "responseType": "ERROR",
                                     "errorCode": 4,
                                     "errorMsg": "The 'store' directory on the transfer node is empty." }
                    else:
                        response = { "responseType": "ERROR",
                                     "errorCode": 5,
                                     "errorMsg": "Service busy. Please, retry later." }
                else:
                    response = { "responseType": "ERROR",
                                 "errorCode": 6,
                                 "errorMsg": "Permission denied." }
        elif requestBody["requestType"] == "STORE_CON":
            username = requestBody["userName"]
            self.prepare(username)
            job = Job()
            job.setType("vos_data")
            job.setInfo(requestBody)
            job.setPhase("PENDING")
            job.setOwnerId(self.dbConn.getUserId(username))
            self.dbConn.insertJob(job)
            dbResponse = self.dbConn.getJob(job.jobId)
            job.jobInfo["storageId"] = requestBody["storageId"]
            job.jobInfo["storageType"] = requestBody["storageType"]
            self.pendingQueueWrite.insertJob(job)
            if "error" in dbResponse:
                response = { "responseType": "ERROR",
                             "errorCode": 7,
                             "errorMsg": "Job creation failed." }
            else:
                response = { "responseType": "STORE_RUN", 
                             "jobId": job.jobId }
        else:
            response = { "responseType": "ERROR",
                         "errorCode": 8,
                         "errorMsg": "Unkown request type." }

        return response

    # TODO
    # to be removed from store_preprocessor.py
    # or simply add a chmod -x here, to be faster?
    def prepare(self, username):
        #self.path = "/home/" + username + "/store"
        path = self.storageStorePath.replace("{username}", username)
        for folder, subfolders, files in os.walk(path):
            os.chown(folder, 0, 0)
            os.chmod(folder, 0o555)
            for s in subfolders:
                os.chown(os.path.join(folder, s), 0, 0)
                os.chmod(os.path.join(folder, s), 0o555)
            for f in files:
                os.chown(os.path.join(folder, f), 0, 0)
                os.chmod(os.path.join(folder, f), 0o555)

    def run(self):
        self.logger.info(f"Starting RPC server of type {self.type}...")
        super(DataRPCServer, self).run()

