#!/usr/bin/env python

import logging
import os

from config import Config
from datetime import datetime as dt
from db_connector import DbConnector
from mailer import Mailer
from node import Node
from redis_log_handler import RedisLogHandler
from system_utils import SystemUtils
from tabulate import tabulate
from task_executor import TaskExecutor


class GroupRwExecutor(TaskExecutor):

    def __init__(self):
        self.type = "group_rw_executor"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  1)
        params = config.loadSection("mail")
        self.adminEmail = params["admin_email"]
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        self.resDir = params["res_dir"]
        self.systemUtils = SystemUtils()
        self.jobObj = None
        self.jobId = None
        self.requestType = None
        self.vospacePath = None
        self.groupName = None
        self.realGroupName = None
        super(GroupRwExecutor, self).__init__()

    def updateGroupRw(self):
        """This method adds/removes groups to/from 'group_read' and 'group_write'."""

        self.dbConn.setPhase(self.jobId, "EXECUTING")
        self.dbConn.setStartTime(self.jobId)
        
        if self.requestType == "GRPR_ADD":
            self.dbConn.updateGroupRead(self.groupName, [], self.vospacePath)
        elif self.requestType == "GRPR_DEL":
            self.dbConn.updateGroupRead([], self.groupName, self.vospacePath)
        elif self.requestType == "GRPW_ADD":
            self.dbConn.updateGroupWrite(self.groupName, [], self.vospacePath)
        elif self.requestType == "GRPW_DEL":
            self.dbConn.updateGroupWrite([], self.groupName, self.vospacePath)
            
        self.jobObj.setPhase("COMPLETED")
        self.dbConn.setPhase(self.jobId, "COMPLETED")
        self.dbConn.setEndTime(self.jobId)
        
        # Send e-mail notification
        m = Mailer()
        m.addRecipient(self.adminEmail)
        if self.requestType == "GRPR_ADD":
            msg = f"""
Added '{self.realGroupName}' to 'group_read' for {self.vospacePath} and any child nodes.
            """
        elif self.requestType == "GRPR_DEL":
            msg = f"""
Removed '{self.realGroupName}' from 'group_read' for {self.vospacePath} and any child nodes.
            """
        elif self.requestType == "GRPW_ADD":
            msg = f"""
Added '{self.realGroupName}' to 'group_write' for {self.vospacePath} and any child nodes.
            """
        elif self.requestType == "GRPW_DEL":
            msg = f"""
Removed '{self.realGroupName}' from 'group_write' for {self.vospacePath} and any child nodes.
            """
        m.setMessage("VOSpace group_rw notification", msg)
        m.send()

    def run(self):
        self.logger.info("Starting group_rw executor...")
        self.setSourceQueueName("group_rw_ready")
        self.setDestinationQueueName("group_rw_terminated")
        while True:
            self.wait()
            if self.srcQueue.len() > 0:
                self.jobObj = self.srcQueue.getJob()
                self.jobId = self.jobObj.jobId
                self.userId = self.jobObj.ownerId
                self.requestType = self.jobObj.jobInfo["requestType"]
                self.vospacePath = self.jobObj.jobInfo["vospacePath"]
                self.groupName = [ self.jobObj.jobInfo["groupName"] ]
                self.realGroupName = self.groupName[0].split("people.")[-1].replace("\\", "")
                self.updateGroupRw()
                if self.destQueue.len() >= self.maxTerminatedJobs:
                    self.destQueue.extractJob()
                self.destQueue.insertJob(self.jobObj)
                self.srcQueue.extractJob()
                self.logger.info(f"Job {self.jobObj.jobId} MOVED from {self.srcQueue.name()} to {self.destQueue.name()}")

