#!/usr/bin/env python

import logging
import os

from redis_log_handler import RedisLogHandler
from redis_rpc_server import RedisRPCServer
from config import Config
from db_connector import DbConnector


class JobRPCServer(RedisRPCServer):

    def __init__(self, host, port, db, rpcQueue):
        self.type = "job"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  2)
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        super(JobRPCServer, self).__init__(host, port, db, rpcQueue)

    def callback(self, requestBody):
        # 'requestType' attribute is mandatory
        if "requestType" not in requestBody:
            response = { "responseType": "ERROR",
                         "errorCode": 1,
                         "errorMsg": "Malformed request, missing parameters." }
        elif requestBody["requestType"] == "JOB_LIST":
            result = self.dbConn.listActiveJobs()
            response = { "responseType": "LST_DONE", "jobList": result }
        elif requestBody["requestType"] == "JOB_BY_PHASE":
            jobPhase = requestBody["jobPhase"]
            result = self.dbConn.listJobsByPhase(jobPhase)
            response = { "responseType": "LST_BY_PHASE_DONE", "jobList": result }
        elif requestBody["requestType"] == "JOB_INFO":
            jobId = requestBody["jobId"]
            if self.dbConn.jobExists(jobId):
                result = self.dbConn.getJobInfo(jobId)
                response = { "responseType": "LST_INFO_DONE", "jobInfo": result }
            else:
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": "Wrong job ID." }
        elif requestBody["requestType"] == "JOB_RESULTS":
            jobId = requestBody["jobId"]
            if self.dbConn.jobExists(jobId):
                result = self.dbConn.getJobResults(jobId)
                response = { "responseType": "LST_RESULTS_DONE", "jobResults": result }
            else:
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": "Wrong job ID." }
        else:
            response = { "responseType": "ERROR",
                         "errorCode": 3,
                         "errorMsg": "Unkown request type." }
        return response

    def run(self):
        self.logger.info(f"Starting RPC server of type {self.type}...")
        super(JobRPCServer, self).run()
