#!/usr/bin/env python

import os
import logging
import shutil
import subprocess
import sys

from config import Config
from db_connector import DbConnector
from redis_log_handler import RedisLogHandler
from system_utils import SystemUtils
from tape_client import TapeClient
from task_executor import TaskExecutor


class StoreExecutor(TaskExecutor):

    def __init__(self):
        self.type = "store_executor"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("spectrum_archive")
        self.tapeClient = TapeClient(params["host"],
                                     params.getint("port"),
                                     params["user"],
                                     params["pkey_file_path"])
        params = config.loadSection("transfer_node")
        self.storageStorePath = params["store_path"]
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  1)
        params = config.loadSection("scheduling")
        self.maxTerminatedJobs = params.getint("max_terminated_jobs")
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        self.jobObj = None
        self.jobId = None
        self.username = None
        self.requestType = None
        self.storageId = None
        self.storageType = None
        self.nodeList = []
        self.systemUtils = SystemUtils()
        super(StoreExecutor, self).__init__()

    def copyData(self):
        self.dbConn.setPhase(self.jobId, "EXECUTING")
        self.dbConn.setStartTime(self.jobId)
        srcPathPrefix = self.storageStorePath.replace("{username}", self.username)
        srcData = os.listdir(srcPathPrefix)
        destPathPrefix = self.dbConn.getStorageBasePath(self.storageId) + '/' + self.username
        sp = subprocess.run(["rsync", "-av", srcPathPrefix + '/', destPathPrefix + '/'], capture_output = True)
        if(sp.returncode or sp.stderr):
            return False
        else:
            return True

    def cleanup(self):
        srcPathPrefix = self.storageStorePath.replace("{username}", self.username)
        srcData = os.listdir(srcPathPrefix)
        for el in srcData:
            nodeOSPath = srcPathPrefix + '/' + el
            if os.path.isdir(nodeOSPath):
                shutil.rmtree(nodeOSPath)
            elif os.path.isfile(nodeOSPath):
                os.remove(nodeOSPath)
            else:
                sys.exit("Unable to remove file/dir on the transfer node!!!")
        userInfo = self.systemUtils.userInfo(self.username)
        uid = userInfo[1]
        gid = userInfo[2]
        os.chown(srcPathPrefix, uid, gid)
        os.chmod(srcPathPrefix, 0o755)

    def update(self):
        out = open("store_executor_log.txt", "a")
        results = [{"target": ""}]                
        self.dbConn.setResults(self.jobId, results)
        for nodeVOSPath in self.nodeList:
            out.write(f"nodeListElement: {nodeVOSPath}\n")
            self.dbConn.setAsyncTrans(nodeVOSPath, True)
            self.dbConn.setJobId(nodeVOSPath, None)
        self.jobObj.setPhase("COMPLETED")
        self.dbConn.setPhase(self.jobId, "COMPLETED")
        self.dbConn.setEndTime(self.jobId)
        out.close()

    def run(self):
        self.logger.info("Starting store executor...")
        self.setSourceQueueName("write_ready")
        self.setDestinationQueueName("write_terminated")
        while True:
            self.wait()
            if self.srcQueue.len() > 0:
                self.jobObj = self.srcQueue.getJob()
                self.jobId = self.jobObj.jobId
                self.username = self.jobObj.jobInfo["userName"]
                self.requestType = self.jobObj.jobInfo["requestType"]
                self.storageId = self.jobObj.jobInfo["storageId"]
                self.storageType = self.jobObj.jobInfo["storageType"]
                self.nodeList = self.jobObj.jobInfo["nodeList"]
                # TODO
                # 1) Controlla il tipo di destinazione: hot (server) o cold (tape)
                # *) HOT
                # 1) Controlla di avere sufficiente spazio su disco sul server
                # 2) Ottieni la cartella o la lista delle cartelle sulla 'store' sul transf. node
                # 3) Avvia la copia delle cartelle con rsync da transf. node a frontend tape
                # *) COLD
                # 1) Controlla che altri non stiano migrando in questo preciso momento (con tape client)
                # 2) Ottieni la cartella o la lista delle cartelle sulla 'store' sul transf. node
                # 3) Avvia la copia delle cartelle con rsync da transf. node a frontend tape
                # 4) A copia finita, se tutto ok, rimuovi i dati sul tn e setta async_trans a true sul db
                self.copyData()
                self.cleanup()
                self.update()
                if self.destQueue.len() == self.maxTerminatedJobs:
                    self.destQueue.extractJob()
                self.destQueue.insertJob(self.jobObj)
                self.srcQueue.extractJob()
                self.logger.info(f"Job {self.jobObj.jobId} MOVED from {self.srcQueue.name()} to {self.destQueue.name()}")
