#!/usr/bin/env python
#
# A FIFO queue based on a Redis list
#
#

import json
import redis

from config import Config
from job import Job


class JobQueue(object):

    def __init__(self, queueName):
        config = Config("/etc/vos_ts/vos_ts.conf")
        self.params = config.loadSection("job_cache")
        self.redisCli = redis.Redis(host = self.params["host"],
                                    port = self.params["port"],
                                    db = self.params["db_sched"])
        self.queueName = queueName

    def len(self):
        """Returns the number of jobs in the current queue."""
        return self.redisCli.llen(self.queueName)

    def name(self):
        """Returns the name of the current queue."""
        return self.queueName

    def getJob(self):
        """Gets a copy of the first job without moving it out from the current queue."""
        job = json.loads(self.redisCli.lrange(self.queueName, self.len() - 1, self.len() - 1)[0].decode("utf-8"))
        jobObj = Job()
        jobObj.setId(job["jobId"])
        jobObj.setType(job["jobType"])
        jobObj.setOwnerId(job["ownerId"])
        jobObj.setPhase(job["phase"])
        jobObj.setQuote(job["quote"])
        jobObj.setExecutionDuration(job["executionDuration"])
        jobObj.setInfo(job["jobInfo"])
        return jobObj

    def insertJob(self, jobObj):
        """Pushes a new job into the queue."""
        data = { "jobId": jobObj.jobId,
                 "jobType": jobObj.type,
                 "ownerId": jobObj.ownerId,
                 "phase": jobObj.phase,
                 "quote": jobObj.quote,
                 "startTime": jobObj.startTime,
                 "endTime": jobObj.endTime,
                 "executionDuration": jobObj.executionDuration,
                 "destruction": jobObj.destruction,
                 "parameters": jobObj.parameters,
                 "results": jobObj.results,
                 "jobInfo": jobObj.jobInfo }
        self.redisCli.lpush(self.queueName, json.dumps(data))

    def extractJob(self):
        """Moves out a job from the end of the current queue."""
        job = json.loads(self.redisCli.brpop(self.queueName)[1].decode("utf-8"))
        jobObj = Job()
        jobObj.setId(job["jobId"])
        jobObj.setType(job["jobType"])
        jobObj.setOwnerId(job["ownerId"])
        jobObj.setPhase(job["phase"])
        jobObj.setQuote(job["quote"])
        jobObj.setExecutionDuration(job["executionDuration"])
        jobObj.setInfo(job["jobInfo"])
        return jobObj

    def moveJobTo(self, nextQueueName):
        """
        Moves out a job from the end of the current queue and places it 
        at the beginning of the next queue (this operation is atomic)
        DEPRECATED
        """
        self.redisCli.brpoplpush(self.queueName, nextQueueName)


# Test
#p = JobQueue("pending")
#r = JobQueue("ready")
#j1 = Job()
#j1.setPhase("PENDING")
#j2 = Job()
#j2.setPhase("PENDING")
#p.insertJob(j1)
#p.insertJob(j2)
#print(p.getJob())
#p.moveJobTo(r.name())
#print(r.getJob())
