#!/usr/bin/env python
#
# TODO:
# - interface with the file catalog (insert)
# - data and permissions cleanup
#

import json
import os
import re
import shutil
import sys

from datetime import datetime as dt

from checksum import Checksum
from config import Config
from file_grouper import FileGrouper
from db_connector import DbConnector
from node import Node
from system_utils import SystemUtils
from task_executor import TaskExecutor


class StorePreprocessor(TaskExecutor):

    def __init__(self):
        self.systemUtils = SystemUtils()
        self.md5calc = Checksum()
        config = Config("/etc/vos_ts/vos_ts.conf")
        self.params = config.loadSection("file_grouper")
        self.fileGrouper = FileGrouper(self.params.getint("min_num_files"),
                                       self.systemUtils.convertSizeToBytes(self.params["max_dir_size"]))
        self.params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(self.params["user"],
                                  self.params["password"],
                                  self.params["host"],
                                  self.params.getint("port"),
                                  self.params["db"],
                                  1,
                                  1)
        self.params = config.loadSection("transfer_node")
        self.storageStorePath = self.params["store_path"]
        self.storageId = None
        self.storageType = None
        self.jobObj = None
        self.jobId = None
        self.username = None
        self.userId = None
        self.nodeList = []
        super(StorePreprocessor, self).__init__()

    def prepare(self, username):
        self.username = username
        self.path = "/home/" + username + "/store"
        for folder, subfolders, files in os.walk(self.path):
            os.chown(folder, 0, 0)
            os.chmod(folder, 0o555)
            for s in subfolders:
                os.chown(os.path.join(folder, s), 0, 0)
                os.chmod(os.path.join(folder, s), 0o555)
            for f in files:
                os.chown(os.path.join(folder, f), 0, 0)
                os.chmod(os.path.join(folder, f), 0o555)

    def execute(self):
        # First scan to find crowded dirs
        [ dirs, files ] = self.systemUtils.scan(self.path)

        # Create a .tar for all dirs matching the constraints, if any
        for dir in dirs:
            self.fileGrouper.recursive(self.path + '/' + dir)

        # Second scan after file grouper execution
        [ dirs, files ] = self.systemUtils.scan(self.path)
        timestamp = dt.now().strftime("%Y_%m_%d-%H_%M_%S")

        # Case 1: /home/user/store contains both files and dirs
        if files and dirs:
            destPath = self.path + '/' + timestamp + "-vos_wrapper"
            try:
                os.mkdir(destPath)
            except OSError as error:
                sys.exit(f"FATAL: {error}")
            for file in files:
                srcPath = self.path + '/' + file
                shutil.move(srcPath, destPath)
            for dir in dirs:
                srcPath = self.path + '/' + dir
                shutil.move(srcPath, destPath)
            self.md5calc.recursive(destPath)
        # Case 2: /home/user/store contains only files
        elif files and not dirs:
            destPath = self.path + '/' + timestamp + "-vos_wrapper"
            try:
                os.mkdir(destPath)
            except OSError as error:
                sys.exit(f"FATAL: {error}")
            for file in files:
                srcPath = self.path + '/' + file
                shutil.move(srcPath, destPath)
            self.md5calc.recursive(destPath)
        # Case 3: /home/user/store contains only dirs
        elif not files and dirs:
            for dir in dirs:
                destPath = self.path + '/' + dir
                self.md5calc.recursive(destPath)
        # Case 4: /home/user/store is empty
        else:
            sys.exit("The 'store' directory is empty.")

        # Third scan after directory structure 'check & repair'
        [ dirs, files ] = self.systemUtils.scanRecursive(self.path)

        # File catalog update
        #out = open("store_preprocessor_log.txt", "a")
        self.userId = self.dbConn.getUserId(self.username)
        #out.write(f"USER: {self.username}\n")
        #out.write(f"USER_ID: {self.userId}\n")
        pathPrefix = self.storageStorePath.replace("{username}", self.username)
        tstampWrapperDirPattern = re.compile("/[0-9]{4}_[0-9]{2}_[0-9]{2}-[0-9]{2}_[0-9]{2}_[0-9]{2}-vos_wrapper")
        for dir in dirs:
            #out.write(f"DIR: {dir}\n")
            #out.write(f"pathPrefix: {pathPrefix}\n")
            basePath = os.path.dirname(dir).replace(pathPrefix, "/" + self.username)
            #out.write(f"basePath: {basePath}\n")
            nodeName = os.path.basename(dir)
            #out.write(f"nodeName: {nodeName}\n")
            cnode = Node(nodeName, "container")
            if not tstampWrapperDirPattern.match("/" + nodeName):
                if tstampWrapperDirPattern.search(basePath):
                    tstampWrapperDir = tstampWrapperDirPattern.search(basePath).group(0).lstrip('/')
                    #out.write(f"tstampWrapperDir: {tstampWrapperDir}\n")
                    basePath = tstampWrapperDirPattern.sub("", basePath)
                    #out.write(f"newBasePath: {basePath}\n")
                    cnode.setWrapperDir(tstampWrapperDir)                
                cnode.setParentPath(basePath)
                locationId = self.dbConn.getLocationId(self.storageId)
                cnode.setLocationId(locationId)
                cnode.setBusyState(True)
                cnode.setCreatorID(self.userId)
                cnode.setContentLength(0)
                cnode.setSticky(True)

                if os.path.islink(dir):
                    # node is a symlink, do not import it...
                    pass
                elif self.dbConn.insertNode(cnode):
                    self.nodeList.append(basePath + '/' + nodeName)
                else:
                    # node already exists, skip it...
                    pass

        #out.write("\n\n")

        for flist in files:
            for file in flist:
                #out.write(f"FILE: {file}\n")
                if self.md5calc.fileIsValid(file):
                    #out.write(f"pathPrefix: {pathPrefix}\n")
                    basePath = os.path.dirname(file).replace(pathPrefix, "/" + self.username)
                    #out.write(f"basePath: {basePath}\n")
                    nodeName = os.path.basename(file)
                    #out.write(f"nodeName: {nodeName}\n")
                    dnode = Node(nodeName, "data")
                    if tstampWrapperDirPattern.search(basePath):
                        tstampWrapperDir = tstampWrapperDirPattern.search(basePath).group(0).lstrip('/')
                        #out.write(f"tstampWrapperDir: {tstampWrapperDir}\n")
                        basePath = tstampWrapperDirPattern.sub("", basePath)
                        #out.write(f"newBasePath: {basePath}\n")
                        dnode.setWrapperDir(tstampWrapperDir)                    
                    dnode.setParentPath(basePath)
                    locationId = self.dbConn.getLocationId(self.storageId)
                    dnode.setLocationId(locationId)
                    dnode.setBusyState(True)
                    dnode.setCreatorID(self.userId)
                    dnode.setContentLength(os.path.getsize(file))
                    dnode.setContentMD5(self.md5calc.getMD5(file))
                    dnode.setSticky(True)

                    if os.path.islink(file):
                        # node is a symlink, do not import it...
                        pass
                    elif self.dbConn.insertNode(dnode):
                        self.nodeList.append(basePath + '/' + nodeName)
                    else:
                        # node already exists, skip it...
                        pass

        #out.write("\n")
        #out.close()
        
    def update(self):
        self.jobObj.setPhase("QUEUED")
        self.dbConn.setPhase(self.jobId, "QUEUED")

    def run(self):
        print("Starting store preprocessor...")
        self.setSourceQueueName("write_pending")
        self.setDestinationQueueName("write_ready")
        while True:
            self.wait()
            if self.destQueue.len() < self.maxReadyJobs and self.srcQueue.len() > 0:
                self.jobObj = self.srcQueue.getJob()
                self.jobId = self.jobObj.jobId
                self.storageId = self.jobObj.jobInfo["storageId"]
                self.storgeType = self.jobObj.jobInfo["storageType"]
                self.username = self.jobObj.jobInfo["userName"]
                self.prepare(self.username)
                self.execute()
                self.update()
                self.jobObj.jobInfo["nodeList"] = self.nodeList.copy()
                self.nodeList.clear()
                self.destQueue.insertJob(self.jobObj)
                self.srcQueue.extractJob()                
                print(f"Job {self.jobObj.jobId} MOVED from {self.srcQueue.name()} to {self.destQueue.name()}")

# Test
#sp = StorePreprocessor()
#sp.prepare("curban")
#sp.execute()
