#!/usr/bin/env python

import os
import re

from amqp_server import AMQPServer
from config import Config
from checksum import Checksum
from db_connector import DbConnector
from node import Node
from system_utils import SystemUtils
from tape_client import TapeClient

from multiprocessing import Process

class ImportAMQPServer(AMQPServer):

    def __init__(self, host, port, queue):
        self.type = "import"
        self.md5calc = Checksum()
        config = Config("/etc/vos_ts/vos_ts.conf")       
        self.params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(self.params["user"], 
                                  self.params["password"], 
                                  self.params["host"], 
                                  self.params.getint("port"), 
                                  self.params["db"],
                                  1,
                                  1)
        self.params = config.loadSection("spectrum_archive")
        self.tapeClient = TapeClient(self.params["host"],
                                     self.params.getint("port"),
                                     self.params["user"],
                                     self.params["pkey_file_path"])
        self.systemUtils = SystemUtils()
        super(ImportAMQPServer, self).__init__(host, port, queue)

    def execute_callback(self, requestBody):
        # 'requestType' and 'path' attributes are mandatory
        if "requestType" not in requestBody or "path" not in requestBody:
            response = { "errorCode": 1, "errorMsg": "Malformed request, missing parameters." }
        elif requestBody["requestType"] == "NODE_IMPORT":
            path = os.path.abspath(requestBody["path"])
            username = requestBody["userName"]
            userInDb = self.dbConn.userExists(username)
            userInfo = self.systemUtils.userInfo(username)
            #out = open("import_amqp_server_log.txt", "a")
            
            if not userInfo or not userInDb:
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": "The user does not exist or is not registered in the database." }
                return response
            
            userId = self.dbConn.getRapId(username)
            pathPrefix = self.dbConn.storageBasePathIsValid(path)
            
            if pathPrefix:
                storageId = self.dbConn.getStorageId(pathPrefix)
                storageType = self.dbConn.getStorageType(pathPrefix)
            else:
                response = { "responseType": "ERROR",
                             "errorCode": 3,
                             "errorMsg": "Invalid storage mount point." }
                return response
            
            if not os.path.exists(path):
                response = { "responseType": "ERROR",
                             "errorCode": 4,
                             "errorMsg": "Path not found." }
                return response
            elif not os.path.isdir(path):
                response = { "responseType": "ERROR",
                             "errorCode": 5,
                             "errorMsg": "Directory path expected." }
                return response
            elif username not in path:
                response = { "responseType": "ERROR",
                             "errorCode": 6,
                             "errorMsg": "Directory path does not contain the username." }
                return response
            elif os.path.dirname(path) != pathPrefix + '/' + username:
                response = { "responseType": "ERROR",
                             "errorCode": 7,
                             "errorMsg": "Invalid path, directory must be located in " + pathPrefix + '/' + username  }
                return response            
            else:
                p = Process(target = self.load, args = (self.tapeClient, self.dbConn, self.md5calc, self.systemUtils, path, pathPrefix, storageType, storageId, userId,), daemon = True)
                p.start()
            # add a counter to track the number of nodes (files and dirs) + log file
            response = { "responseType": "IMPORT_DONE" }
        else:
            response = { "responseType": "ERROR",
                         "errorCode": 8,
                         "errorMsg": "Unkown request type." }

        return response

    def run(self):
        print(f"Starting AMQP server of type {self.type}...")
        super(ImportAMQPServer, self).run()
        
    def load(self, tapeClient, dbConn, md5calc, systemUtils, path, pathPrefix, storageType, storageId, userId):
        out = open("import_amqp_server_log.txt", "a")
        if storageType == "cold":
                tapeClient.connect()
                tapeClient.recallChecksumFiles(path)
                tapeClient.disconnect()
                
        [ dirs, files ] = systemUtils.scanRecursive(os.path.dirname(path))
                
        tstampWrapperDirPattern = re.compile("/[0-9]{4}_[0-9]{2}_[0-9]{2}-[0-9]{2}_[0-9]{2}_[0-9]{2}-vos_wrapper")
        for dir in dirs:            
            out.write(f"DIR dir: {dir}\n")
            out.write(f"DIR pathPrefix: {pathPrefix}\n\n")
                    
            if path in dir:
                parentPath = os.path.dirname(dir).split(pathPrefix)[1]
                nodeName = os.path.basename(dir)

                cnode = Node(nodeName, "container")
                        
                if not tstampWrapperDirPattern.match("/" + nodeName):
                    if tstampWrapperDirPattern.search(parentPath):
                        tstampWrapperDir = tstampWrapperDirPattern.search(parentPath).group(0).lstrip('/')
                        parentPath = tstampWrapperDirPattern.sub("", parentPath)
                        cnode.setWrapperDir(tstampWrapperDir)
                                
                    if parentPath == '/':
                        vospacePath = parentPath + nodeName
                    else:
                        vospacePath = parentPath + '/' + nodeName
                        
                    cnode.setParentPath(parentPath)
                    locationId = dbConn.getLocationId(storageId)
                    cnode.setLocationId(locationId)
                    cnode.setOwnerID(userId)
                    cnode.setCreatorID(userId)
                    cnode.setContentLength(0)
                    if not dbConn.nodeExists(cnode):
                        dbConn.insertNode(cnode)
                        dbConn.setAsyncTrans(vospacePath, True)
                        dbConn.setSticky(vospacePath, True)

        for flist in files:
            for file in flist:                
                if md5calc.fileIsValid(file) and path in os.path.dirname(file):
                    out.write(f"FILE files: {files}\n")
                    out.write(f"FILE flist: {flist}\n")
                    out.write(f"FILE file: {file}\n")
                    out.write(f"FILE pathPrefix: {pathPrefix}\n")
                    parentPath = os.path.dirname(file).split(pathPrefix)[1]
                    out.write(f"FILE parentPath: {parentPath}\n")
                    nodeName = os.path.basename(file)
                    out.write(f"FILE nodeName: {nodeName}\n")                                                    
                    dnode = Node(nodeName, "data")
                            
                    if tstampWrapperDirPattern.search(parentPath):
                        tstampWrapperDir = tstampWrapperDirPattern.search(parentPath).group(0).lstrip('/')
                        parentPath = tstampWrapperDirPattern.sub("", parentPath)
                        dnode.setWrapperDir(tstampWrapperDir)
                            
                    vospacePath = parentPath + '/' + nodeName
                    out.write(f"FILE vospacePath: {vospacePath}\n")
                    dnode.setParentPath(parentPath)
                    storageId = dbConn.getStorageId(pathPrefix)
                    locationId = dbConn.getLocationId(storageId)
                    dnode.setLocationId(locationId)
                    dnode.setOwnerID(userId)
                    dnode.setCreatorID(userId)
                    dnode.setContentLength(os.path.getsize(file))
                    dnode.setContentMD5(md5calc.getMD5(file))
                            
                    if not dbConn.nodeExists(dnode):
                        dbConn.insertNode(dnode)
                        dbConn.setAsyncTrans(vospacePath, True)
                        dbConn.setSticky(vospacePath, True)

