#!/usr/bin/env python

import os

from config import Config
from db_connector import DbConnector
from job_queue import JobQueue
from job import Job
from redis_rpc_server import RedisRPCServer
from system_utils import SystemUtils


class ImportRPCServer(RedisRPCServer):

    def __init__(self, host, port, db, rpcQueue):
        self.type = "import"
        config = Config("/etc/vos_ts/vos_ts.conf")
        self.params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(self.params["user"],
                                  self.params["password"],
                                  self.params["host"],
                                  self.params.getint("port"),
                                  self.params["db"],
                                  1,
                                  1)
        self.params = config.loadSection("scheduling")
        self.maxReadyJobs = self.params.getint("max_ready_jobs")
        self.importReadyQueue = JobQueue("import_ready")
        self.systemUtils = SystemUtils()
        super(ImportRPCServer, self).__init__(host, port, db, rpcQueue)

    def callback(self, requestBody):
        # 'requestType' and 'path' attributes are mandatory
        if "requestType" not in requestBody or "path" not in requestBody:
            response = { "responseType": "ERROR",
                         "errorCode": 1,
                         "errorMsg": "Malformed request, missing parameters." }
        elif requestBody["requestType"] == "NODE_IMPORT":
            path = os.path.abspath(requestBody["path"])
            username = requestBody["userName"]
            userInDb = self.dbConn.userExists(username)
            userInfo = self.systemUtils.userInfo(username)
            #out = open("import_amqp_server_log.txt", "a")

            if not userInfo or not userInDb:
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": "The user does not exist or is not registered in the database." }
                return response

            userId = self.dbConn.getUserId(username)
            pathPrefix = self.dbConn.storageBasePathIsValid(path)

            if pathPrefix:
                storageId = self.dbConn.getStorageId(pathPrefix)
                storageType = self.dbConn.getStorageType(pathPrefix)
            else:
                response = { "responseType": "ERROR",
                             "errorCode": 3,
                             "errorMsg": "Invalid storage mount point." }
                return response

            if not os.path.exists(path):
                response = { "responseType": "ERROR",
                             "errorCode": 4,
                             "errorMsg": "Path not found." }
            elif not os.path.isdir(path):
                response = { "responseType": "ERROR",
                             "errorCode": 5,
                             "errorMsg": "Directory path expected." }
            #elif username not in path:
            #    response = { "responseType": "ERROR",
            #                 "errorCode": 6,
            #                 "errorMsg": "Directory path does not contain the username." }
            #elif os.path.dirname(path) != pathPrefix + '/' + username:
            #    response = { "responseType": "ERROR",
            #                 "errorCode": 7,
            #                 "errorMsg": "Invalid path, directory must be located in " + pathPrefix + '/' + username  }
            elif self.importReadyQueue.len() >= self.maxReadyJobs:
                response = { "responseType": "ERROR",
                             "errorCode": 8,
                             "errorMsg": "Import queue is full, please, retry later." }
            else:
                jobObj = Job()
                jobObj.setType("vos_import")
                jobInfo = requestBody.copy()
                jobInfo["pathPrefix"] = pathPrefix
                jobInfo["storageId"] = storageId
                jobInfo["storageType"] = storageType
                jobObj.setInfo(jobInfo)
                jobObj.setPhase("QUEUED")
                jobObj.setOwnerId(userId)
                self.dbConn.insertJob(jobObj)
                self.importReadyQueue.insertJob(jobObj)

            response = { "responseType": "IMPORT_STARTED" }
        else:
            response = { "responseType": "ERROR",
                         "errorCode": 9,
                         "errorMsg": "Unkown request type." }

        return response

    def run(self):
        print(f"Starting RPC server of type {self.type}...")
        super(ImportRPCServer, self).run()
