#!/usr/bin/env python

import hashlib
import os
import sys

from config import Config
from system_utils import SystemUtils


class Checksum(object):

    def __init__(self):
        self.systemUtils = SystemUtils()
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("checksum")
        self.fileBufferSize = self.systemUtils.convertSizeToBytes(params["file_buffer_size"])
        self.md5FileSuffix = params["md5_file_suffix"]

    def setFileBufferSize(fileBufferSize):
        """Sets the buffer size in bytes when reading a chunk of data."""
        self.fileBufferSize = fileBufferSize

    def fileIsValid(self, absFilePath):
        """Checks whether a file is a checksum file or not"""
        if not self.md5FileSuffix in absFilePath:
            return True
        else:
            return False

    def getMD5(self, absFilePath):
        """
        Returns the MD5 checksum corresponding to a give filename
        according to its absolute path on disk.
        """
        if self.fileIsValid(absFilePath):
            md5FileName = os.path.dirname(absFilePath) + self.md5FileSuffix
            md5File = open(md5FileName, "r")
            for row in md5File:
                md5sum = row.split("  ./")[0]
                fileName = row.split("  ./")[1].rstrip()
                if fileName == os.path.basename(absFilePath):
                    return md5sum
        return None

    def md5sum(self, filePath):
        """Calculates the MD5 checksum of a file."""
        md5Hash = hashlib.md5()
        with open(filePath, "rb") as f:
            for chunk in iter(lambda: f.read(self.fileBufferSize), b""):
                md5Hash.update(chunk)
        return md5Hash.hexdigest()

    def recursive(self, folderPath):
        """
        Calculates MD5 checksums recursively and saves on disk the list of
        files in a folder with also the corresponding checksums.
        """
        for folder, subfolder, files in os.walk(folderPath, topdown = False):
            cwd = os.path.basename(folder)
            parent = os.path.dirname(folder)
            md5FilePath = os.path.abspath(parent) + '/' + cwd + self.md5FileSuffix
            if not os.path.exists(md5FilePath):
                if not files == []:
                    md5file = open(md5FilePath, 'w')
                    for file in files:
                        filePath = os.path.abspath(folder) + '/' + file
                        md5file.write(self.md5sum(filePath) + "  ./" + file + '\n')
                    md5file.close()
