#!/usr/bin/env python

import logging

from config import Config
from db_connector import DbConnector
from job_queue import JobQueue
from job import Job
from redis_log_handler import RedisLogHandler
from redis_rpc_server import RedisRPCServer


class GroupRwRPCServer(RedisRPCServer):

    def __init__(self, host, port, db, rpcQueue):
        self.type = "group_rw"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  2)
        params = config.loadSection("scheduling")
        self.maxReadyJobs = params.getint("max_ready_jobs")
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        self.groupRwReadyQueue = JobQueue("group_rw_ready")
        super(GroupRwRPCServer, self).__init__(host, port, db, rpcQueue)

    def callback(self, requestBody):
        # 'requestType' and 'path' attributes are mandatory
        if "requestType" not in requestBody or "vospacePath" not in requestBody:
            response = { "responseType": "ERROR",
                         "errorCode": 1,
                         "errorMsg": "Malformed request, missing parameters." }
        elif (requestBody["requestType"] == "GRPR_ADD" or 
              requestBody["requestType"] == "GRPR_DEL" or
              requestBody["requestType"] == "GRPW_ADD" or 
              requestBody["requestType"] == "GRPW_DEL"):
            requestType = requestBody["requestType"]
            jobType = requestType.split('_')[0]
            vospacePath = requestBody["vospacePath"]
            groupname = requestBody["groupName"]            

            if not self.dbConn.nodeExists(vospacePath):
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": "VOSpace path not found." }
            elif self.groupRwReadyQueue.len() >= self.maxReadyJobs:
                response = { "responseType": "ERROR",
                             "errorCode": 3,
                             "errorMsg": "'group_rw_ready' queue is full, please, retry later." }
            else:
                jobObj = Job()
                jobObj.setType(f"vos_{jobType.lower()}")
                jobInfo = requestBody.copy()
                jobObj.setInfo(jobInfo)
                jobObj.setPhase("QUEUED")
                creatorId = self.dbConn.getCreatorId(vospacePath)
                jobObj.setOwnerId(creatorId)
                self.dbConn.insertJob(jobObj)
                self.groupRwReadyQueue.insertJob(jobObj)
                response = { "responseType": f"{jobType}_STARTED" }
        elif requestBody["requestType"] == "GRPR_LST":
            vospacePath = requestBody["vospacePath"]            
            if not self.dbConn.nodeExists(vospacePath):
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": "VOSpace path not found." }
            else:
                result = self.dbConn.getGroupRead(vospacePath)

                response = { "responseType": "GRPR_LST_DONE",
                             "groupList": result }
        elif requestBody["requestType"] == "GRPW_LST":
            vospacePath = requestBody["vospacePath"]
            if not self.dbConn.nodeExists(vospacePath):
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": "VOSpace path not found." }
            else:
                result = self.dbConn.getGroupWrite(vospacePath)
                response = { "responseType": "GRPW_LST_DONE",
                             "groupList": result }
        else:
            response = { "responseType": "ERROR",
                         "errorCode": 4,
                         "errorMsg": "Unkown request type." }

        return response

    def run(self):
        self.logger.info(f"Starting RPC server of type {self.type}...")
        super(GroupRwRPCServer, self).run()
 
