#!/usr/bin/env python

import logging
import mimetypes
import smtplib

from config import Config
from email.message import EmailMessage
from email.policy import SMTP
from redis_log_handler import RedisLogHandler
from smtplib import SMTPConnectError
from smtplib import SMTPException


class Mailer(object):

    def __init__(self):
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("mail")
        self.smtpServer = params["smtp_server"]
        self.smtpPort = params.getint("smtp_port")
        self.sender = params["no_reply_email"]
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        self.recipients = []
        self.message = None

    def sender(self, sender):
        """Set mail sender."""
        self.sender = sender

    def addRecipient(self, recipient):
        """Add recipient to recipients list."""
        self.recipients.append(recipient)

    def setMessage(self, subject, msg):
        """Set a basic message."""
        self.message = EmailMessage()
        self.message["From"] = self.sender
        self.message["Subject"] = subject
        self.message["To"] = ", ".join(self.recipients)
        self.message.set_content(msg)

    def setMessageWithAttachment(self, subject, msg, filePath):
        """Set a message with attachment."""
        self.setMessage(subject, msg)
        ctype, encoding = mimetypes.guess_type(filePath)
        if ctype is None or encoding is not None:
            ctype = "application/octet-stream"
        maintype, subtype = ctype.split('/', 1)

        with open(filePath, "rb") as fp:
            self.message.add_attachment(fp.read(),
                                        maintype = maintype,
                                        subtype = subtype,
                                        filename = filePath)

    def send(self):
        """Send email message."""
        try:
            smtpObj = smtplib.SMTP(self.smtpServer, self.smtpPort)
            smtpObj.send_message(self.message)
            logMsg = "E-mail message sent successfully!"
            self.logger.debug(logMsg)
        except SMTPConnectError:
            logMsg = "Cannot connect to SMTP server."
            self.logger.exception(logMsg)
        except TimeoutError:
            logMsg = "Connection timeout."
            self.logger.exception(logMsg)
        except SMTPException:
            logMsg = "Cannot send email message."
            self.logger.exception(logMsg)


# Test 1
#m1 = Mailer()
#m1.addRecipient("cristiano.urban@inaf.it")
#m1.setMessage("Hello", "Hello world!")
#m1.send()

# Test 2
#m2 = Mailer()
#m2.addRecipient("cristiano.urban@inaf.it")
#m2.addRecipient("cristiano.urban.slack@gmail.com")
#m2.setMessageWithAttachment("Hello", "Hello world!", "config.py")
#m2.send()
