#!/usr/bin/env python

import logging
import mimetypes
import os
import smtplib

from config import Config
from email.message import EmailMessage
from email.policy import SMTP
from redis_log_handler import RedisLogHandler
from smtplib import SMTPConnectError
from smtplib import SMTPException


class Mailer(object):

    def __init__(self, logger):
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("mail")
        self.smtpServer = params["smtp_server"]
        self.smtpPort = params.getint("smtp_port")
        self.sender = params["no_reply_email"]
        self.enabled = params.getboolean("enable_notifications")
        self.logger = logger
        self.recipients = []
        self.message = None

    def sender(self, sender):
        """Set mail sender."""
        self.sender = sender

    def addRecipient(self, recipient):
        """Add recipient to recipients list."""
        self.recipients.append(recipient)

    def setMessage(self, subject, msg):
        """Set a basic message."""
        self.message = EmailMessage()
        self.message["From"] = self.sender
        self.message["Subject"] = subject
        self.message["To"] = ", ".join(self.recipients)
        self.message.set_content(msg)

    def setMessageWithAttachment(self, subject, msg, filePath):
        """Set a message with attachment."""
        self.setMessage(subject, msg)
        ctype, encoding = mimetypes.guess_type(filePath)
        if ctype is None or encoding is not None:
            ctype = "application/octet-stream"
        maintype, subtype = ctype.split('/', 1)

        with open(filePath, "rb") as fp:
            self.message.add_attachment(fp.read(),
                                        maintype = maintype,
                                        subtype = subtype,
                                        filename = os.path.basename(filePath))

    def send(self):
        """Send email message."""
        if self.enabled:
            self.logger.debug("E-mail notifications enabled.")
            try:
                smtpObj = smtplib.SMTP(self.smtpServer, self.smtpPort)
                smtpObj.send_message(self.message)
            except SMTPConnectError:
                self.logger.exception("Cannot connect to SMTP server.")
            except TimeoutError:
                self.logger.exception("Connection timeout.")
            except SMTPException:
                self.logger.exception("Cannot send email message.")
            else:
                self.logger.debug("E-mail message sent successfully!")
        else:
            self.logger.debug("E-mail notifications disabled.")


# Test 1
#m1 = Mailer()
#m1.addRecipient("cristiano.urban@inaf.it")
#m1.setMessage("Hello", "Hello world!")
#m1.send()

# Test 2
#m2 = Mailer()
#m2.addRecipient("cristiano.urban@inaf.it")
#m2.addRecipient("cristiano.urban.slack@gmail.com")
#m2.setMessageWithAttachment("Hello", "Hello world!", "config.py")
#m2.send()
