#!/usr/bin/env python

import os
import logging
import shutil
import subprocess
import sys

from config import Config
from datetime import datetime as dt
from db_connector import DbConnector
from mailer import Mailer
from redis_log_handler import RedisLogHandler
from system_utils import SystemUtils
from tabulate import tabulate
from tape_client import TapeClient
from task_executor import TaskExecutor


class StoreExecutor(TaskExecutor):
    
    # We use 10 GB of tolerance when we calculate
    # the free space in our storage point
    TOL = 10 * (2**30)

    def __init__(self):
        self.type = "store_executor"
        self.systemUtils = SystemUtils()
        config = Config("/etc/vos_ts/vos_ts.conf")        
        params = config.loadSection("transfer_node")
        self.storageStorePath = params["store_path"]
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  1)
        params = config.loadSection("transfer")
        self.maxBlockSize = self.systemUtils.convertSizeToBytes(params["block_size"])
        params = config.loadSection("scheduling")
        self.maxTerminatedJobs = params.getint("max_terminated_jobs")
        params = config.loadSection("mail")
        self.adminEmail = params["admin_email"]
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        self.resDir = params["res_dir"]
        params = config.loadSection("spectrum_archive")
        self.tapePool = params["tape_pool"]
        self.tapeClient = TapeClient(params["host"],
                                     params.getint("port"),
                                     params["user"],
                                     params["pkey_file_path"],
                                     self.logger)
        self.jobObj = None
        self.jobId = None
        self.username = None
        self.userId = None
        self.userEmail = None
        self.requestType = None
        self.storageId = None
        self.storageType = None
        self.nodeList = []
        super(StoreExecutor, self).__init__()

    def copyData(self):
        self.dbConn.setPhase(self.jobId, "EXECUTING")
        self.dbConn.setStartTime(self.jobId)
        srcPathPrefix = self.storageStorePath.replace("{username}", self.username)
        srcData = os.listdir(srcPathPrefix)
        self.logger.debug("Checking storage available space...")
        storageBasePath = self.dbConn.getStorageBasePath(self.storageId)
        storageFreeSpace = 0
        tapeFrontendFreeSpace = 0
        if self.storageType == "hot":
            [ total, used, free ] = self.systemUtils.getFileSystemSize(storageBasePath)
            storageFreeSpace = free - self.TOL
            self.logger.debug(f"storageFreeSpace (hot): {storageFreeSpace} B")
            if storageFreeSpace < self.dataSize:
                return False
        else:
            #[ total, used, free ] = self.systemUtils.getFileSystemSize(storageBasePath)
            #tapeFrontendFreeSpace = free - self.maxBlockSize - self.TOL
            self.tapeClient.connect()
            tapePoolList = self.tapeClient.getPoolList()
            for el in tapePoolList:
                if el.getName() == self.tapePool:
                    storageFreeSpace = el.getFreeSpace() - self.TOL
            self.tapeClient.disconnect()
            #self.logger.debug(f"tapeFrontendFreeSpace (cold): {tapeFrontendFreeSpace}")
            self.logger.debug(f"storageFreeSpace (cold): {storageFreeSpace} B")
            #if tapeFrontendFreeSpace < self.dataSize or storageFreeSpace < self.dataSize:
            if storageFreeSpace < self.dataSize:
                return False
        destPathPrefix = storageBasePath + '/' + self.username
        sp = subprocess.run(["rsync", "-av", srcPathPrefix + '/', destPathPrefix + '/'], capture_output = True)
        if(sp.returncode or sp.stderr):
            return False
        else:
            return True

    def cleanup(self):
        srcPathPrefix = self.storageStorePath.replace("{username}", self.username)
        srcData = os.listdir(srcPathPrefix)
        for el in srcData:
            nodeOSPath = srcPathPrefix + '/' + el
            if os.path.isdir(nodeOSPath):
                shutil.rmtree(nodeOSPath)
            elif os.path.isfile(nodeOSPath):
                os.remove(nodeOSPath)
            else:
                sys.exit("Unable to remove file/dir on the transfer node!!!")
        userInfo = self.systemUtils.userInfo(self.username)
        uid = userInfo[1]
        gid = userInfo[2]
        os.chown(srcPathPrefix, uid, gid)
        os.chmod(srcPathPrefix, 0o755)

    def update(self, status):
        results = [{"target": ""}]                
        self.dbConn.setResults(self.jobId, results)
        
        if status == "OK":
            for el in self.nodeList:
                nodeVOSPath = el[2]
                self.dbConn.setAsyncTrans(nodeVOSPath, True)
                self.dbConn.setJobId(nodeVOSPath, None)
        
            timestamp = dt.now().strftime("%Y_%m_%d-%H_%M_%S")
            nodeListFile = os.path.join(self.resDir, "vos_data_report-" + timestamp)
            nlfp = open(nodeListFile, "w")
            nlfp.write(tabulate(self.nodeList,
                                headers = [ "Timestamp", "OS path", "VOSpace path", "Node type", "Result"],
                                tablefmt = "simple"))
            nlfp.close()
        
            self.jobObj.setPhase("COMPLETED")
            self.dbConn.setPhase(self.jobId, "COMPLETED")
            self.dbConn.setEndTime(self.jobId)
        
            # Send e-mail notification
            m = Mailer(self.logger)
            m.addRecipient(self.adminEmail)
            self.userEmail = self.dbConn.getUserEmail(self.userId)
            if self.userEmail != self.adminEmail:
                m.addRecipient(self.userEmail)
            msg = f"""
        [VOSpace data storage procedure summary]

        Storage type: {self.storageType}
        Storage ID: {self.storageId}
        Creator ID: {self.userId}
        Processed nodes: {len(self.nodeList)}
        Stored nodes: {sum(res[-1] == 'DONE' for res in self.nodeList)}
        Skipped nodes: {sum(res[-1] == 'SKIPPED' for res in self.nodeList)}
        Symlinks detected: {sum(res[-1] == 'SYMLINK' for res in self.nodeList)}

        """
        
            if len(self.nodeList) <= 10 ** 5:            
                m.setMessageWithAttachment("VOSpace data storage notification: Job COMPLETED", msg, nodeListFile)
            else:
                info = f"""
        INFO: 
        this operation involved a number of nodes greater than 10^5,
        you will find the results in {self.resDir}.

            """
                msg += info
                m.setMessage("VOSpace data storage notification", msg)
            m.send()
        else:
            self.jobObj.setPhase("ERROR")
            self.dbConn.setPhase(self.jobId, "ERROR")
            self.dbConn.setEndTime(self.jobId)
            
            # Send e-mail notification
            m = Mailer(self.logger)
            m.addRecipient(self.adminEmail)
            self.userEmail = self.dbConn.getUserEmail(self.userId)
            if self.userEmail != self.adminEmail:
                m.addRecipient(self.userEmail)
            msg = f"""
        [VOSpace data storage procedure summary]

        Storage type: {self.storageType}
        Storage ID: {self.storageId}
        Creator ID: {self.userId}
        """
        
            info = f"""
        INFO: 
        the job was terminated due to an error that occurred 
        while copying the data on the storage point.
        
        """
            msg += info
            m.setMessage("VOSpace data storage notification: Job ERROR", msg)        
        self.nodeList.clear()

    def run(self):
        self.logger.info("Starting store executor...")
        self.setSourceQueueName("write_ready")
        self.setDestinationQueueName("write_terminated")
        while True:
            self.wait()
            if self.srcQueue.len() > 0:
                self.jobObj = self.srcQueue.getJob()
                self.jobId = self.jobObj.jobId
                self.username = self.jobObj.jobInfo["userName"]
                self.userId = self.jobObj.ownerId
                self.requestType = self.jobObj.jobInfo["requestType"]
                self.storageId = self.jobObj.jobInfo["storageId"]
                self.storageType = self.jobObj.jobInfo["storageType"]
                self.nodeList = self.jobObj.jobInfo["nodeList"].copy()
                self.dataSize = self.jobObj.jobInfo["dataSize"]
                self.logger.debug(f"dataSize: {self.dataSize} B")
                # TODO
                # 1) Controlla il tipo di destinazione: hot (server) o cold (tape)
                # *) HOT
                # 1) Controlla di avere sufficiente spazio su disco sul server
                # 2) Ottieni la cartella o la lista delle cartelle sulla 'store' sul transf. node
                # 3) Avvia la copia delle cartelle con rsync da transf. node a frontend tape
                # *) COLD
                # 1) Controlla che altri non stiano migrando in questo preciso momento (con tape client)
                # 2) Ottieni la cartella o la lista delle cartelle sulla 'store' sul transf. node
                # 3) Avvia la copia delle cartelle con rsync da transf. node a frontend tape
                # 4) A copia finita, se tutto ok, rimuovi i dati sul tn e setta async_trans a true sul db                
                if self.copyData():
                    self.cleanup()
                    self.update("OK")
                else:
                    self.update("ERROR")
                if self.destQueue.len() >= self.maxTerminatedJobs:
                    self.destQueue.extractJob()
                self.destQueue.insertJob(self.jobObj)
                self.srcQueue.extractJob()
                self.logger.info(f"Job {self.jobObj.jobId} MOVED from '{self.srcQueue.name()}' to '{self.destQueue.name()}'")
