#!/usr/bin/env python
#
# This file is part of vospace-transfer-service
# Copyright (C) 2021 Istituto Nazionale di Astrofisica
# SPDX-License-Identifier: GPL-3.0-or-later
#

import datetime
import json
import logging

from config import Config
from db_connector import DbConnector
from mailer import Mailer
from redis_log_handler import RedisLogHandler
from task_executor import TaskExecutor


class JobMonitor(TaskExecutor):

    def __init__(self):
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  1)
        params = config.loadSection("monitoring")
        self.delayTime = params.getint("delay_time")
        self.maxJobDuration = json.loads(params["max_job_duration"])
        params = config.loadSection("mail")
        self.adminEmail = params["admin_email"]
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.jobTimeoutDays = self.maxJobDuration["days"]
        self.jobTimeoutSeconds = self.maxJobDuration["hours"] * 3600 + self.maxJobDuration["minutes"] * 60 + self.maxJobDuration["seconds"]
        self.executingJobList = []
        self.longJobs = []
        self.notifiedJobs = []
        super(JobMonitor, self).__init__()
    
    def execute(self):
        try:
            try:
                self.executingJobList = self.dbConn.getJobsByPhase("EXECUTING")
            except Exception:
                self.logger.exception("FATAL: unable to obtain info about executing jobs.")
                return False
            now = datetime.datetime.now()
            for job in self.executingJobList:
                delta = now - datetime.datetime.fromisoformat(job["start_time"])
                if delta.days >= self.jobTimeoutDays and delta.seconds > self.jobTimeoutSeconds:
                    self.longJobs.append(job["job_id"])
            # Notify the user and write the log file only if new jobs get stuck
            if self.longJobs and self.longJobs != self.notifiedJobs:
                m = Mailer(self.logger)
                m.addRecipient(self.adminEmail)
                msg = f"The following jobs are taking too long to complete, jobIDs = {self.longJobs}"
                self.logger.warning(msg)
                m.setMessage(f"VOSpace job alert notification", msg)
                m.send()
                self.notifiedJobs.clear()
                self.notifiedJobs = self.longJobs.copy()
            return True
        except Exception:
            self.logger.exception("FATAL: something went wrong during the execution phase.")
            return False
        
    def cleanup(self):
        self.executingJobList.clear()
        self.longJobs.clear()
    
    def run(self):
        self.logger.info("Starting job monitor...")
        while True:
            self.wait(delayTime = self.delayTime)
            self.execute()
            self.cleanup()
