#!/usr/bin/env python
#
# This file is part of vospace-transfer-service
# Copyright (C) 2021 Istituto Nazionale di Astrofisica
# SPDX-License-Identifier: GPL-3.0-or-later
#

import datetime
import json
import logging
import os
import shutil

from config import Config
from db_connector import DbConnector
from redis_log_handler import RedisLogHandler
from task_executor import TaskExecutor


class RetrieveCleaner(TaskExecutor):

    def __init__(self):
        self.type = "retrieve_cleaner"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("transfer_node")
        self.storageRetrievePath = params["retrieve_path"]
        params = config.loadSection("async_recall")
        self.filesExpiration = json.loads(params["files_expiration"])
        self.days = self.filesExpiration["days"]
        self.seconds = self.filesExpiration["hours"] * 3600 + self.filesExpiration["minutes"] * 60 + self.filesExpiration["seconds"]
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  1,
                                  self.logger)
        self.jobObj = None
        self.username = None
        self.nodeList = []
        self.destPathList = []
        super(RetrieveCleaner, self).__init__()
        
    def dataHasExpired(self):
        # Avoid "all zero" condition
        if self.days <= 0 and self.seconds < 30:
            self.days = 0
            self.seconds = 30
        elif self.seconds >= 86400:
            self.days += self.seconds // 86400
            self.seconds = self.seconds % 86400
        
        jobEndTime = datetime.datetime.fromisoformat(self.jobObj.endTime)
        currentTime = datetime.datetime.now()
        delta = currentTime - jobEndTime
        if delta.days >= self.days and delta.seconds > self.seconds:
            return True
        else:
            return False
        
    def execute(self):
        # while dim lists > 0:
        #   loop over the two lists (nodeList and destPathList):
        #     if the vospace node is not busy: 
        #       set 'async_trans' = True
        #       delete the file/dir in the 'retrieve' directory
        #       remove the corresponding elements on the two lists
        try:
            self.logger.info("++++++++++ Start of execution phase ++++++++++")
            try:
                nodeInfo = self.dbConn.getOSPath(self.nodeList[0])
                self.username = nodeInfo["username"]
            except Exception:
                self.logger.exception("FATAL: unable to obtain info about VOSpace nodes.")
                return False
            numNodes = len(self.nodeList)
            self.logger.info(f"Number of VOSpace nodes involved: {numNodes}")
            while numNodes > 0:                    
                i = 0
                while i < numNodes:
                    vospacePath = self.nodeList[i]
                    destPath = self.destPathList[i]
                    try:
                        busy = self.dbConn.nodeIsBusy(vospacePath)
                    except Exception:
                        self.logger.exception(f"FATAL: unable to check the 'busy' flag value for the VOSpace node '{vospacePath}'.")
                        return False
                    if not busy:
                        try:
                            self.dbConn.setAsyncTrans(vospacePath, True)
                        except Exception:
                            self.logger.exception(f"FATAL: unable to update the 'async_trans' flag for the VOSpace node '{vospacePath}'.")
                            return False
                        try:
                            if os.path.isfile(destPath):
                                os.remove(destPath)
                            else:
                                shutil.rmtree(destPath)
                        except FileNotFoundError:
                            self.logger.exception(f"Cannot find '{destPath}', skip...")
                        # check for empty dirs and remove them
                        basePath = self.storageRetrievePath.replace("{username}", self.username)
                        for root, dirs, files in os.walk(basePath, topdown = False):
                            for dir in dirs:
                                dirPath = os.path.abspath(root) + '/' + dir
                                if not os.listdir(dirPath):
                                    os.rmdir(dirPath)
                        self.nodeList.pop(i)
                        self.destPathList.pop(i)
                        numNodes -= 1
                        i = 0
        except Exception:
            self.logger.exception("FATAL: something went wrong while cleaning the expired data.")
            return False
        else:
            self.logger.info("++++++++++ End of execution phase ++++++++++")
            return True

    def cleanup(self):
        self.nodeList.clear()
        self.destPathList.clear()
        
    def run(self):
        self.logger.info("Starting retrieve cleaner...")
        self.setSourceQueueName("read_terminated")
        self.setDestinationQueueName("read_clean")
        while True:
            self.wait()
            try:
                srcQueueLen = self.srcQueue.len()
                destQueueLen = self.destQueue.len()
            except Exception:
                self.logger.exception("Cache error: failed to retrieve queue length.")
            else:
                if srcQueueLen > 0:
                    self.jobObj = self.srcQueue.getJob()
                    self.nodeList = self.jobObj.nodeList.copy()
                    self.destPathList = self.jobObj.jobInfo["destPathList"].copy()
                    if self.dataHasExpired():
                        if self.execute():
                            try:
                                if destQueueLen >= self.maxCleanJobs:
                                    self.destQueue.extractJob()
                                self.destQueue.insertJob(self.jobObj)
                                self.srcQueue.extractJob()
                            except Exception:
                                self.logger.exception(f"Failed to move job {self.jobObj.jobId} from '{self.srcQueue.name()}' to '{self.destQueue.name()}'")
                            else:
                                self.logger.info(f"Job {self.jobObj.jobId} MOVED from '{self.srcQueue.name()}' to '{self.destQueue.name()}'")
                    self.cleanup()
