#!/usr/bin/env python
#
# This file is part of vospace-transfer-service
# Copyright (C) 2021 Istituto Nazionale di Astrofisica
# SPDX-License-Identifier: GPL-3.0-or-later
#

import os
import re
import shutil
import sys

from config import Config


class SystemUtils(object):

    UNITS = {
                "B": 1,
                "KB": 1 << 10,
                "MB": 1 << 20,
                "GB": 1 << 30,
                "TB": 1 << 40,
                "PB": 1 << 50,
            }

    def __init__(self):
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("transfer_node")
        self.vospaceUserBasePath = params["base_path"]


    def userInfo(self, username):
        """
        Obtains username, UID and GID associated to a given user
        from the user directory (name.surname) on the transfer node.
        """
        try:
            vospaceUserFolderPath = self.vospaceUserBasePath.replace("{username}", username)
            uid = os.stat(vospaceUserFolderPath).st_uid
            gid = os.stat(vospaceUserFolderPath).st_gid
        except OSError:
            return False
        else:
            return [ username, uid, gid ]

    def findIllegalCharacters(self, name):
        """Checks for file/dir names containing illegal characters."""
        pattern = re.compile(r"[<>?\":/|'*`\\]")
        if re.search(pattern, name):
            return True
        else:
            return False

    def findInvalidFileAndDirNames(self, path):
        """
        Scans 'path' recursively to search for file/dir names with
        invalid characters.
        """
        invalidList = []
        for folder, subfolders, files in os.walk(path, topdown = True):
            cwd = os.path.basename(folder)
            folderAbsPath = os.path.dirname(folder) + '/' + cwd
            if self.findIllegalCharacters(cwd):
                invalidList.append([folderAbsPath])
            for f in files:
                fileAbsPath = folderAbsPath + '/' + f
                if self.findIllegalCharacters(f):
                    invalidList.append([fileAbsPath])
        return invalidList

    def scanRecursive(self, path):
        """Performs a recursive scan of dirs and files."""
        dirList = []
        fileList = []
        for folder, subfolders, files in os.walk(path, topdown = True):
            cwd = os.path.basename(folder)
            if folder != path:
                folderAbsPath = os.path.dirname(folder) + '/' + cwd
                dirList.append(folderAbsPath)
                i = 0
                for f in files:
                    fileAbsPath = folderAbsPath + '/' + f
                    files[i] = fileAbsPath
                    i += 1
                fileList.append(files)
        return [ dirList, fileList ]

    def scan(self, path):
        """
        Performs a scan of dirs and files only at the first level,
        removing symlinks, if any.
        """
        dirList = []
        fileList = []
        elementList = os.listdir(path)
        for el in elementList:
            elPath = path + '/' + el
            if os.path.islink(elPath):
                os.remove(elPath)
            elif os.path.isdir(elPath):
                dirList.append(el)
            elif os.path.isfile(elPath):
                fileList.append(el)
            else:
                # do nothing...
                pass
        return [ dirList, fileList ]

    def getSize(self, path):
        """
        If 'path' is a file returns the file size in bytes,
        if 'path' is a directory it returns the total size of the dir,
        in all the other cases it returns 0
        """
        size = 0
        if os.path.isfile(path) and not os.path.islink(path):
            size = os.path.getsize(path)
        elif os.path.isdir(path) and not os.path.islink(path):
            for folder, subfolders, files in os.walk(path, topdown = True):
                cwd = os.path.basename(folder)
                parent = os.path.dirname(folder)
                base = parent + '/' + cwd
                if not os.path.islink(folder):
                    for f in files:
                        file = base + '/' + f
                        if not os.path.islink(file):
                            size += os.path.getsize(file)
        return size

    def getFileSystemSize(self, path):
        """
        Returns a named tuple with usage statistcs in the format
        (total, used, free):
        - total: total space available in bytes
        - used: used space in bytes
        - free: free space in bytes
        """
        return shutil.disk_usage(path)

    def convertSizeToBytes(self, sizeStr):
        size = int(sizeStr.split()[0])
        unit = sizeStr.split()[1]
        if unit not in self.UNITS:
            return False
        else:
            return size * self.UNITS[unit]


# Test
#sysUtils = SystemUtils()
#info = sysUtils.userInfo("cristiano")
#print(sysUtils.convertSizeInBytes(" 10  MB  "))
#print(info)
#size = sysUtils.getSize("/home/cristiano/EDC/Books/")
#print(f"size: {size / (2**20)}")
#[total, used, free] = sysUtils.getFileSystemSize("/")
#print(f"total: {total}, used: {used}, free: {free}")
