/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.Job;
import java.util.List;
import java.util.function.Consumer;
import net.ivoa.xml.uws.v1.ErrorSummary;
import net.ivoa.xml.uws.v1.ErrorType;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import org.springframework.web.client.RestTemplate;

@SpringBootTest
@AutoConfigureMockMvc
@TestPropertySource(properties = {"vospace-authority=example.com!vospace"})
public class JobControllerTest {

    @MockBean
    private VOSpaceClient client;

    @MockBean
    private RestTemplate restTemplate;

    @Autowired
    private MockMvc mockMvc;

    @Test
    public void testSingleFileAsyncRecall() throws Exception {

        JobSummary job = new JobSummary();
        job.setPhase(ExecutionPhase.QUEUED);

        when(client.startTransferJob(any(), any())).thenReturn(job);

        mockMvc.perform(post("/recall")
                .contentType(MediaType.APPLICATION_JSON)
                .content("[\"/path/to/file\"]"))
                .andExpect(status().isOk());
    }

    @Test
    public void testMultipleFilesAsyncRecall() throws Exception {

        JobSummary job = new JobSummary();
        job.setPhase(ExecutionPhase.PENDING);

        when(client.startTransferJob(argThat(transfer -> {
            assertEquals(2, transfer.getView().getParam().size());
            assertEquals("vos://example.com!vospace/path/to", transfer.getTarget());
            assertEquals("file1", transfer.getView().getParam().get(0).getValue());
            assertEquals("file2", transfer.getView().getParam().get(1).getValue());
            return true;
        }), any())).thenReturn(job);

        when(client.getFileServiceEndpoint(any(), any())).thenReturn("http://file-service/path/to/file");

        mockMvc.perform(post("/recall")
                .contentType(MediaType.APPLICATION_JSON)
                .content("[\"/path/to/file1\", \"/path/to/file2\"]"))
                .andExpect(status().isOk());
    }

    @Test
    public void testAsyncRecallErrorWithDetails() throws Exception {

        JobSummary job = new JobSummary();
        job.setPhase(ExecutionPhase.ERROR);

        when(client.getErrorDetail(any(), any())).thenReturn("Error was xxx");

        ErrorSummary errorSummary = new ErrorSummary();
        errorSummary.setHasDetail(true);
        errorSummary.setType(ErrorType.FATAL);

        job.setErrorSummary(errorSummary);

        when(client.startTransferJob(any(), any())).thenReturn(job);

        testErrorCall(ex -> assertTrue(ex.getMessage().contains("Error was xxx")));
    }

    @Test
    public void testAsyncRecallTransientErrorWithoutDetails() throws Exception {

        JobSummary job = new JobSummary();
        job.setPhase(ExecutionPhase.ERROR);

        ErrorSummary errorSummary = new ErrorSummary();
        errorSummary.setMessage("simple message.");
        errorSummary.setType(ErrorType.TRANSIENT);

        job.setErrorSummary(errorSummary);

        when(client.startTransferJob(any(), any())).thenReturn(job);

        testErrorCall(ex -> assertTrue(ex.getMessage().contains("simple message") && ex.getMessage().contains("retry")));
    }

    @Test
    public void testAsyncRecallErrorWithoutSummary() throws Exception {

        JobSummary job = new JobSummary();
        job.setPhase(ExecutionPhase.ERROR);

        when(client.startTransferJob(any(), any())).thenReturn(job);

        testErrorCall(ex -> assertTrue(ex.getMessage().contains("error")));
    }

    @Test
    public void testGetJobs() throws Exception {

        when(client.getAsyncRecallJobs(any())).thenReturn(List.of(new Job(new JobSummary(), Job.JobType.ASYNC_RECALL)));
        when(client.getArchiveJobs(any())).thenReturn(List.of(new Job(new JobSummary(), Job.JobType.ARCHIVE)));

        mockMvc.perform(get("/jobs"))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$").isArray())
                .andExpect(jsonPath("$[0].type").value("ASYNC_RECALL"))
                .andExpect(jsonPath("$[1].type").value("ARCHIVE"));
    }

    private void testErrorCall(Consumer<Exception> exceptionChecker) throws Exception {
        try {
            mockMvc.perform(post("/recall")
                    .contentType(MediaType.APPLICATION_JSON)
                    .content("[\"/path/to/file\"]"))
                    .andExpect(status().is5xxServerError());
        } catch (Exception ex) {
            exceptionChecker.accept(ex);
            return;
        }
        fail("An error was expected");
    }
}
