/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.vospace.ui.TokenProvider;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.CreateLinkRequest;
import it.inaf.ia2.vospace.ui.exception.BadRequestException;
import it.inaf.ia2.vospace.ui.exception.VOSpaceStatusException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

@RestController
public class CreateLinksController extends BaseController {

    private static final Logger LOG = LoggerFactory.getLogger(CreateLinksController.class);

    @Value("${list-of-links.limit:1000}")
    private int listOfLinksSizeLimit;

    @Autowired
    private VOSpaceClient client;

    @Autowired
    private TokenProvider tokenProvider;

    @Autowired
    private Executor requestsExecutor;

    @PostMapping(value = "/createLink", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<?> createLink(@RequestBody CreateLinkRequest request) {

        Optional<String> token = tokenProvider.getToken();

        ContainerNode parent = getFolder(request.getFolder(), token);

        String uri = parent.getUri() + "/" + request.getNodeName();

        LinkNode link = new LinkNode();
        link.setUri(uri);
        link.setTarget(request.getUrl());

        client.createNode(link, token);

        return ResponseEntity.noContent().build();
    }

    @PostMapping(value = "/uploadLinks", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public ResponseEntity<?> uploadLinks(@RequestParam(value = "file", required = true) MultipartFile file,
            @RequestParam("folder") String folder) throws IOException {

        Optional<String> token = tokenProvider.getToken();

        ContainerNode parent = getFolder(folder, token);

        String fileContent = new String(file.getBytes());

        // Execute HTTP calls for links creation in bunches of 20 calls performed in parallel
        List<List<CompletableFuture<?>>> httpCallsGroups = new ArrayList<>();
        List<CompletableFuture<?>> currentHttpCallsGroup = new ArrayList<>();
        httpCallsGroups.add(currentHttpCallsGroup);

        // normalize newlines and split on them
        String[] urls = fileContent.replaceAll("\\r\\n?", "\n").split("\n");
        if (urls.length > listOfLinksSizeLimit) {
            throw new BadRequestException("List is too large: " + urls.length + " lines detected, limit is " + listOfLinksSizeLimit);
        }

        for (String url : urls) {
            if (!url.isBlank()) {

                url = url.trim();
                String uri = parent.getUri() + "/" + getFileNameFromUrl(url);

                LinkNode link = new LinkNode();
                link.setUri(uri);
                link.setTarget(url);

                if (currentHttpCallsGroup.size() > 20) {
                    currentHttpCallsGroup = new ArrayList<>();
                    httpCallsGroups.add(currentHttpCallsGroup);
                }

                currentHttpCallsGroup.add(CompletableFuture.supplyAsync(() -> client.createNode(link, token), requestsExecutor));
            }
        }

        for (List<CompletableFuture<?>> httpCallsGroup : httpCallsGroups) {
            CompletableFuture.allOf(httpCallsGroup.toArray(CompletableFuture[]::new)).join();
        }

        return ResponseEntity.noContent().build();
    }

    private String getFileNameFromUrl(String url) {

        try {
            // parse URL and remove the query string
            String urlPath = new URL(url).getPath();
            if (urlPath.endsWith("/")) {
                // remove last char if it is a slash
                urlPath = urlPath.substring(0, urlPath.length() - 1);
            }
            if (urlPath.isEmpty() || !urlPath.contains("/")) {
                throw new BadRequestException("Unable to extract file name from URL " + url);
            }
            String fileName = urlPath.substring(urlPath.lastIndexOf("/") + 1);
            if (fileName.isEmpty()) {
                throw new BadRequestException("Unable to extract file name from URL " + url);
            }
            return fileName;
        } catch (MalformedURLException ex) {
            throw new BadRequestException("Invalid URL: " + url);
        }
    }

    private ContainerNode getFolder(String folderPath, Optional<String> token) {
        try {
            return (ContainerNode) client.getNode("/" + folderPath, token);
        } catch (VOSpaceStatusException ex) {
            if (ex.getHttpStatus() == 404) {
                throw new BadRequestException("Folder parameter specified a non-existent folder: /" + folderPath);
            }
            throw ex;
        }
    }
}
