/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.vospace.ui.TokenProvider;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.Job;
import it.inaf.ia2.vospace.ui.exception.BadRequestException;
import it.inaf.oats.vospace.datamodel.Views;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import net.ivoa.xml.uws.v1.ErrorType;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.vospace.v2.Param;
import net.ivoa.xml.vospace.v2.Transfer;
import net.ivoa.xml.vospace.v2.View;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class JobController extends BaseController {

    private static final Logger LOG = LoggerFactory.getLogger(JobController.class);

    @Value("${vospace-authority}")
    private String authority;

    @Autowired
    private VOSpaceClient client;

    @Autowired
    private TokenProvider tokenProvider;

    @Autowired
    private Executor requestsExecutor;

    @PostMapping(value = "/recall", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Job> startRecallFromTapeJob(@RequestBody List<String> paths) {

        if (paths.isEmpty()) {
            throw new BadRequestException("Received empty list of nodes");
        }

        Transfer transfer = new Transfer();
        transfer.setDirection("pullToVoSpace");

        View view = new View();
        view.setUri(Views.ASYNC_RECALL_VIEW_URI);
        transfer.setView(view);

        if (paths.size() == 1) {
            transfer.setTarget("vos://" + authority + paths.get(0));
        } else {
            String parent = paths.get(0).substring(0, paths.get(0).lastIndexOf("/"));
            transfer.setTarget("vos://" + authority + parent);
            for (String path : paths) {
                Param param = new Param();
                param.setUri(Views.ASYNC_RECALL_VIEW_URI + "/include");
                param.setValue(path.substring(parent.length() + 1));
                view.getParam().add(param);
            }
        }

        Optional<String> token = tokenProvider.getToken();

        JobSummary job = client.startTransferJob(transfer, token);

        if (job.getPhase() == ExecutionPhase.QUEUED
                || job.getPhase() == ExecutionPhase.PENDING
                || job.getPhase() == ExecutionPhase.EXECUTING) {
            return ResponseEntity.ok(new Job(job, Job.JobType.ASYNC_RECALL));
        }
        String errorMessage;
        if (job.getPhase() == ExecutionPhase.ERROR) {
            if (job.getErrorSummary() != null) {
                if (job.getErrorSummary().isHasDetail()) {
                    errorMessage = client.getErrorDetail(job.getJobId(), token);
                } else {
                    errorMessage = job.getErrorSummary().getMessage();
                }
                if (job.getErrorSummary().getType() == ErrorType.TRANSIENT) {
                    if (!errorMessage.endsWith(".")) {
                        errorMessage += ".";
                    }
                    errorMessage += " Please retry later.";
                }
            } else {
                errorMessage = "Unexpected error";
            }
        } else {
            LOG.error("Error while executing job {}", job.getJobId());
            errorMessage = "Error while executing job. Job phase is "
                    + job.getPhase() + ". QUEUED, PENDING or EXECUTING expected";
        }
        throw new RuntimeException(errorMessage);
    }

    @GetMapping(value = "/jobs", produces = MediaType.APPLICATION_JSON_VALUE)
    public List<Job> getJobs() throws Exception {

        Optional<String> token = tokenProvider.getToken();

        CompletableFuture<List<Job>> asyncRecallJobsCall
                = CompletableFuture.supplyAsync(() -> client.getAsyncRecallJobs(token), requestsExecutor);

        CompletableFuture<List<Job>> archiveJobsCall
                = CompletableFuture.supplyAsync(() -> client.getArchiveJobs(token), requestsExecutor);

        CompletableFuture.allOf(asyncRecallJobsCall, archiveJobsCall).join();

        List<Job> jobs = new ArrayList<>();
        jobs.addAll(asyncRecallJobsCall.get());
        jobs.addAll(archiveJobsCall.get());

        return jobs;
    }
}
