/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.aa.data.User;
import static it.inaf.oats.vospace.datamodel.NodeUtils.urlEncodePath;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import net.ivoa.xml.vospace.v2.Node;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;

public class MainNodesHtmlGenerator extends NodesHtmlGenerator {

    public MainNodesHtmlGenerator(Node node, User user, String authority, List<Node> linkedNodes) {
        super(node, user, authority, linkedNodes);
    }

    @Override
    protected Element createContainerElement(Document html) {
        Element container = html.body().appendElement("tbody");
        container.attr("id", "nodes");
        return container;
    }

    @Override
    protected void addChild(Node child, Element containerElement) {

        Node linkedNode = getLinkedNode(child);

        NodeInfo nodeInfo = new NodeInfo(child, user, authority, linkedNode);

        Element row = containerElement.appendElement("tr");

        addSelectionCell(nodeInfo, row);
        addLinkCell(nodeInfo, row);
        addSizeCell(nodeInfo, row);
        addGroupReadCell(nodeInfo, row);
        addGroupWriteCell(nodeInfo, row);
        addActionsCell(nodeInfo, row);
    }

    private void addSelectionCell(NodeInfo nodeInfo, Element row) {

        Element cell = row.appendElement("td");

        Element input = cell.appendElement("input");
        input.attr("type", "checkbox");
        input.attr("data-node", nodeInfo.getPath());

        if (nodeInfo.isAsyncTrans()) {
            input.addClass("async");
        } else if (nodeInfo.isDeletable()) {
            input.addClass("deletable");
        }
    }

    private void addLinkCell(NodeInfo nodeInfo, Element row) {

        Element cell = row.appendElement("td");
        addNodeIcon(nodeInfo, cell);
        addLink(nodeInfo, cell);
    }

    private void addSizeCell(NodeInfo nodeInfo, Element row) {
        Element cell = row.appendElement("td");
        cell.text(nodeInfo.getSize());
    }

    private void addGroupReadCell(NodeInfo nodeInfo, Element row) {
        Element cell = row.appendElement("td");
        fillGroupCell(cell, nodeInfo.getGroupRead());
    }

    private void addGroupWriteCell(NodeInfo nodeInfo, Element row) {
        Element cell = row.appendElement("td");
        fillGroupCell(cell, nodeInfo.getGroupWrite());
    }

    private void fillGroupCell(Element cell, String groups) {
        String[] values = groups.split(" ");
        List<String> personGroups = new ArrayList<>();
        List<String> peopleGroups = new ArrayList<>();
        for (String value : values) {
            if (!value.isBlank()) {
                if (value.startsWith("people.")) {
                    personGroups.add(value.substring("people.".length()).replace("\\.", "."));
                } else {
                    peopleGroups.add(value);
                }
            }
        }
        if (!personGroups.isEmpty()) {
            Element personIcon = cell.appendElement("span");
            personIcon.attr("class", "icon person-icon");
            cell.appendText(String.join(" ", personGroups));
            cell.append("&nbsp;");
        }
        if (!peopleGroups.isEmpty()) {
            Element personIcon = cell.appendElement("span");
            personIcon.attr("class", "icon people-icon");
            cell.appendText(String.join(" ", peopleGroups));
        }
    }

    private void addActionsCell(NodeInfo nodeInfo, Element row) {
        Element cell = row.appendElement("td");

        Element dotsMenu = cell.appendElement("span");
        dotsMenu.attr("class", "dots-menu icon dots-menu-icon pointer");

        Element dropdown = dotsMenu.appendElement("span");
        dropdown.attr("class", "dots-menu-content dropdown-menu");

        String nodePathJs = makeJsArg(nodeInfo.getPath());

        if (nodeInfo.isWritable()) {
            Element shareBtn = dropdown.appendElement("button");
            shareBtn.text("Share");
            shareBtn.attr("type", "button");
            shareBtn.attr("class", "dropdown-item");
            shareBtn.attr("onclick", "shareNode(" + nodePathJs
                    + "," + makeJsArg(nodeInfo.getGroupRead())
                    + "," + makeJsArg(nodeInfo.getGroupWrite()) + ")");
        }
        if (nodeInfo.isDeletable()) {
            Element renameBtn = dropdown.appendElement("button");
            renameBtn.text("Rename");
            renameBtn.attr("type", "button");
            renameBtn.attr("class", "dropdown-item");
            renameBtn.attr("onclick", "renameNode(" + nodePathJs + ")");

            Element moveBtn = dropdown.appendElement("button");
            moveBtn.text("Move");
            moveBtn.attr("type", "button");
            moveBtn.attr("class", "dropdown-item");
            moveBtn.attr("onclick", "moveNode(" + nodePathJs + ")");

            Element copyBtn = dropdown.appendElement("button");
            copyBtn.text("Copy");
            copyBtn.attr("type", "button");
            copyBtn.attr("class", "dropdown-item");
            copyBtn.attr("onclick", "copyNode(" + nodePathJs + ")");

            Element deleteBtn = dropdown.appendElement("button");
            deleteBtn.text("Delete");
            deleteBtn.attr("type", "button");
            deleteBtn.attr("class", "dropdown-item");
            deleteBtn.attr("onclick", "deleteNode(" + nodePathJs + ")");
        }
    }

    private void addLink(NodeInfo nodeInfo, Element cell) {
        if (isDownloadable(nodeInfo, user)) {
            Element link = cell.appendElement("a");
            String href;
            if (nodeInfo.isFolder()) {
                String path = nodeInfo.isLink() ? nodeInfo.getTarget() : nodeInfo.getPath();
                href = "#/nodes" + urlEncodePath(path);
            } else {
                href = "download" + urlEncodePath(nodeInfo.getPath());
                link.attr("target", "blank_");
            }
            link.attr("href", href);
            link.text(nodeInfo.getName());
        } else {
            cell.appendText(nodeInfo.getName());
        }
    }

    private boolean isDownloadable(NodeInfo nodeInfo, User user) {
        if (nodeInfo.isBrokenLink()) {
            return false;
        }

        if (nodeInfo.isFile()) {
            if (nodeInfo.isAsyncTrans() || nodeInfo.isBusy()) {
                return false;
            }
        }
        if (nodeInfo.isPublic()) {
            return true;
        }

        if (nodeInfo.getCreator().equals(user.getName())) {
            return true;
        }

        if (user.getGroups() != null) {
            if (user.getGroups().contains("VOSpace.ADMIN")) {
                return true;
            }
            if (!user.getGroups().isEmpty() && !nodeInfo.getGroupRead().isEmpty()) {
                List<String> groupRead = Arrays.asList(nodeInfo.getGroupRead().split(" "));
                for (String group : groupRead) {
                    if (user.getGroups().contains(group)) {
                        return true;
                    }
                }
            }
        }

        return false;
    }
}
