/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.gms.client.GmsClient;
import it.inaf.ia2.rap.client.ClientCredentialsRapClient;
import it.inaf.ia2.rap.data.AccessTokenResponse;
import it.inaf.ia2.rap.data.Identity;
import it.inaf.ia2.rap.data.IdentityType;
import it.inaf.ia2.rap.data.RapUser;
import it.inaf.ia2.vospace.ui.TokenProvider;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.ShareRequest;
import it.inaf.ia2.vospace.ui.data.SharingInfo;
import it.inaf.ia2.vospace.ui.exception.VOSpaceStatusException;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import net.ivoa.xml.vospace.v2.Property;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.jupiter.MockitoExtension;

@ExtendWith(MockitoExtension.class)
public class SharingServiceTest {

    @Mock
    private GmsClient gmsClient;

    @Mock
    private ClientCredentialsRapClient rapClient;

    @Mock
    private HttpServletRequest servletRequest;

    @Mock
    private VOSpaceClient vospaceClient;

    @Mock
    private TokenProvider tokenProvider;

    @InjectMocks
    private SharingService sharingService;

    @BeforeEach
    public void setUp() {
        sharingService.trustedEppnScope = "inaf.it";
        sharingService.authority = "example.com!vospace";

        AccessTokenResponse tokenResponse = new AccessTokenResponse();
        tokenResponse.setAccessToken("<admin-token>");

        when(rapClient.getAccessTokenFromClientCredentials()).thenReturn(tokenResponse);

        when(rapClient.getUsers(anyString(), any())).thenReturn(getRapUsers());

        when(gmsClient.listGroups(eq(""), eq(true))).thenReturn(
                Arrays.asList("group1", "group2", "people.mario\\.rossi", "people.john\\.doe"));
    }

    @Test
    public void testGetSharingInfo() {

        SharingInfo sharingInfo = sharingService.getSharingInfo();

        assertEquals(2, sharingInfo.getGroups().size());
        assertTrue(sharingInfo.getGroups().contains("group1"));
        assertTrue(sharingInfo.getGroups().contains("group2"));

        assertEquals(4, sharingInfo.getPeople().size());
        assertTrue(sharingInfo.getPeople().contains("mario.rossi"));
        assertTrue(sharingInfo.getPeople().contains("bianca.verdi"));
        assertTrue(sharingInfo.getPeople().contains("paolo.gialli"));
        assertTrue(sharingInfo.getPeople().contains("john.doe"));
    }

    @Test
    public void testSetNodeGroups() {

        DataNode node = new DataNode();
        node.setUri("vos://example.com!vospace/mynode");
        Property groupReadProperty = new Property();
        groupReadProperty.setUri(NodeProperties.GROUP_READ_URI);
        groupReadProperty.setValue("group1");
        node.getProperties().add(groupReadProperty);

        when(vospaceClient.getNode(any(), any())).thenReturn(node);

        ShareRequest shareRequest = new ShareRequest();
        shareRequest.setPath("/mynode");
        shareRequest.setUserRead(Arrays.asList("bianca.verdi", "john.doe"));
        shareRequest.setGroupRead(Arrays.asList("group1", "group2"));
        shareRequest.setUserWrite(Arrays.asList("bianca.verdi"));
        shareRequest.setGroupWrite(Arrays.asList("group2"));

        sharingService.setNodeGroups(shareRequest);

        verify(gmsClient, times(1)).createGroup(eq("people.bianca\\.verdi"), eq(true));

        verify(vospaceClient, times(1)).setNode(argThat(n -> {
            List<String> groupRead = NodeProperties.getNodePropertyAsListByURI(n, NodeProperties.GROUP_READ_URI);
            assertEquals(4, groupRead.size());
            assertTrue(groupRead.contains("people.bianca\\.verdi"));
            assertTrue(groupRead.contains("people.john\\.doe"));
            assertTrue(groupRead.contains("group1"));
            assertTrue(groupRead.contains("group2"));

            List<String> groupWrite = NodeProperties.getNodePropertyAsListByURI(n, NodeProperties.GROUP_WRITE_URI);
            assertEquals(2, groupWrite.size());
            assertTrue(groupWrite.contains("people.bianca\\.verdi"));
            assertTrue(groupWrite.contains("group2"));
            return true;
        }), anyBoolean(), any());
    }

    @Test
    public void testCreateSharedFolder() {

        DataNode node = new DataNode();
        node.setUri("vos://example.com!vospace/anna.bianchi/mynode");

        when(vospaceClient.getNode(any(), any())).thenReturn(node);

        ShareRequest shareRequest = new ShareRequest();
        shareRequest.setPath("/anna.bianchi/mynode");
        shareRequest.setUserRead(List.of("mario.rossi", "paolo.gialli", "bianca.verdi"));
        shareRequest.setUserWrite(List.of());
        shareRequest.setGroupRead(List.of());
        shareRequest.setGroupWrite(List.of());
        shareRequest.setNewPeople(List.of("mario.rossi", "paolo.gialli", "bianca.verdi"));

        ContainerNode marioHome = new ContainerNode();
        marioHome.setUri("vos://example.com!vospace/mario.rossi");
        ContainerNode biancaHome = new ContainerNode();
        biancaHome.setUri("vos://example.com!vospace/bianca.verdi");

        // Mario and Bianca have their home, Paolo not
        when(vospaceClient.getNode(eq("/mario.rossi"), adminToken())).thenReturn(marioHome);
        when(vospaceClient.getNode(eq("/bianca.verdi"), adminToken())).thenReturn(biancaHome);
        doThrow(new VOSpaceStatusException("Not found", 404)).when(vospaceClient).getNode(eq("/paolo.gialli"), any());

        // Mario's Shared Files folder has to be created
        ContainerNode marioShared = new ContainerNode();
        marioShared.setUri("vos://example.com!vospace/mario.rossi/Shared%20Files");
        when(vospaceClient.createNode(argThat(n -> n.getUri().equals(marioShared.getUri())), adminToken())).thenReturn(marioShared);

        // Bianca's Shared Files folder already exists
        doThrow(new VOSpaceStatusException("Conflict", 409)).when(vospaceClient).createNode(
                argThat(n -> n.getUri().equals("vos://example.com!vospace/bianca.verdi/Shared%20Files")), any());

        doAnswer(inv -> inv.getArgument(0)).when(vospaceClient).createNode(argThat(n -> {
            return n instanceof LinkNode
                    && ((LinkNode) n).getTarget().equals("vos://example.com!vospace/anna.bianchi/mynode");
        }), adminToken());

        // Share node
        sharingService.setNodeGroups(shareRequest);

        // Check Mario's shared link
        verify(vospaceClient, times(1)).createNode(argThat(n -> {
            return n.getUri().startsWith("vos://example.com!vospace/mario.rossi/Shared%20Files/mynode");
        }), adminToken());

        // Check Bianca's shared link
        verify(vospaceClient, times(1)).createNode(argThat(n -> {
            return n.getUri().startsWith("vos://example.com!vospace/bianca.verdi/Shared%20Files/mynode");
        }), adminToken());
    }

    private Optional<String> adminToken() {
        return argThat(op -> ((Optional) op).isPresent() && "<admin-token>".equals(((Optional) op).get()));
    }

    private List<RapUser> getRapUsers() {
        List<RapUser> users = new ArrayList<>();

        users.add(getRapUser("1", "mario.rossi"));
        users.add(getRapUser("2", "bianca.verdi"));
        users.add(getRapUser("3", "paolo.gialli"));

        RapUser ignored = getRapUser("4", "name.surname");
        ignored.getIdentities().get(0).setType(IdentityType.GOOGLE);
        users.add(ignored);

        return users;
    }

    private RapUser getRapUser(String id, String name) {
        RapUser user = new RapUser();
        user.setId(id);
        List<Identity> identities = new ArrayList<>();
        Identity identity = new Identity();
        identity.setType(IdentityType.EDU_GAIN);
        identity.setEppn(name + "@inaf.it");
        identities.add(identity);
        user.setIdentities(identities);
        return user;
    }
}
