package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.ListNodeData;
import it.inaf.ia2.vospace.ui.service.NodesService;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class NodesController extends BaseController {

    private static final Logger LOG = LoggerFactory.getLogger(NodesController.class);
    
    @Value("${vospace-authority}")
    private String authority;

    @Autowired
    private NodesService nodesService;

    @Autowired
    private VOSpaceClient client;

    @Autowired
    private HttpServletRequest servletRequest;

    @GetMapping(value = {"/nodes", "/nodes/**"}, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<ListNodeData> listNodes(User principal) throws Exception {

        String path = getPath("/nodes/");
        LOG.debug("listNodes called for path {}", path);

        return ResponseEntity.ok(nodesService.generateNodesHtml(path, principal));
    }

    @DeleteMapping(value = {"/nodes", "/nodes/**"})
    public void deleteNode() {
        String path = getPath("/nodes/");
        LOG.debug("deleteNode called for path {}", path);
        client.deleteNode(path);
    }

    @GetMapping(value = "/download/**")
    public ResponseEntity<?> directDownload() {

        String path = getPath("/download/");
        LOG.debug("directDownload called for path {}", path);

        Transfer transfer = new Transfer();
        transfer.setDirection("pullFromVoSpace");
        transfer.setTarget("vos://" + authority + path);

        Protocol protocol = new Protocol();
        protocol.setUri("ivo://ivoa.net/vospace/core#httpget");
        transfer.getProtocols().add(protocol);

        String url = client.getFileServiceEndpoints(transfer).get(0).getEndpoint();
        HttpHeaders headers = new HttpHeaders();
        headers.set("Location", url);
        return new ResponseEntity<>(headers, HttpStatus.SEE_OTHER);
    }

    @PostMapping(value = "/folder")
    public void newFolder(@RequestBody Map<String, Object> params) {

        String parentPath = getRequiredParam(params, "parentPath");
        if (!parentPath.startsWith("/")) {
            parentPath = "/" + parentPath;
        }
        String name = getRequiredParam(params, "name");
        
        LOG.debug("newFolder called for path {}/{}", parentPath, name);

        ContainerNode node = new ContainerNode();
        node.setUri("vos://" + authority + parentPath + "/" + name);

        Property creator = new Property();
        creator.setUri("ivo://ivoa.net/vospace/core#creator");
        creator.setValue(getUser().getName());
        node.getProperties().add(creator);

        client.createNode(node);
    }

    protected String getPath(String prefix) {
        String requestURL = servletRequest.getRequestURL().toString();
        return NodeUtils.getPathFromRequestURLString(requestURL, prefix);
    }
}
