/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.client;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.vospace.ui.VOSpaceUiApplication;
import it.inaf.ia2.vospace.ui.data.Job;
import it.inaf.ia2.vospace.ui.exception.BadRequestException;
import it.inaf.ia2.vospace.ui.exception.VOSpaceStatusException;
import it.inaf.ia2.vospace.ui.exception.VOSpaceException;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import static it.inaf.oats.vospace.datamodel.NodeUtils.urlEncodePath;
import it.inaf.oats.vospace.datamodel.collections.NodeCollection;
import it.inaf.oats.vospace.datamodel.collections.NodeCollectionsWrapper;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import java.net.ConnectException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Scanner;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ForkJoinPool;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.xml.bind.JAXB;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.uws.v1.Jobs;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Component
public class VOSpaceClient {

    private static final Logger LOG = LoggerFactory.getLogger(VOSpaceClient.class);

    @Value("${use-json}")
    private boolean useJson;

    private static final ObjectMapper MAPPER = new ObjectMapper();

    private final HttpClient httpClient;
    private final String baseUrl;
    private final ForkJoinPool jaxbExecutor;

    public VOSpaceClient(@Value("${vospace-backend-url}") String backendUrl) {
        if (backendUrl.endsWith("/")) {
            // Remove final slash from configured URL
            backendUrl = backendUrl.substring(0, backendUrl.length() - 1);
        }
        baseUrl = backendUrl;

        jaxbExecutor = VOSpaceUiApplication.getJaxbExecutor();

        httpClient = HttpClient.newBuilder()
                .followRedirects(HttpClient.Redirect.ALWAYS)
                .version(HttpClient.Version.HTTP_1_1)
                .build();
    }

    public Node getNode(String path, Optional<String> token) {

        HttpRequest request = getRequest("/nodes" + urlEncodePath(path), token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> unmarshal(res, Node.class));
    }

    public NodeCollectionsWrapper getNodeCollections(Optional<String> token) {

        HttpRequest request = getRequest("/collections", token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .GET()
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200,
                res -> unmarshal(res, NodeCollectionsWrapper.class));
    }

    public JobSummary startTransferJob(Transfer transfer, Optional<String> token) {

        HttpRequest request = getRequest("/transfers?PHASE=RUN", token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .POST(HttpRequest.BodyPublishers.ofString(marshal(transfer)))
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> unmarshal(res, JobSummary.class));
    }

    public String getFileServiceEndpoint(Transfer transfer, Optional<String> token) {

        HttpRequest request = getRequest("/synctrans", token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .POST(HttpRequest.BodyPublishers.ofString(marshal(transfer)))
                .build();

        List<Protocol> protocols = new ArrayList<>();

        HttpResponse<InputStream> redirectResponse = call(request, BodyHandlers.ofInputStream(), 200, (res, previousRes) -> {
            Transfer transferRes = unmarshal(res, Transfer.class);
            protocols.addAll(transferRes.getProtocols());
            // IMPORTANT: HTTP call for checking the error status (in case of empty protocols list) must be
            // performed outside this lambda otherwise the exception "IllegalStateException: No thread-bound request found"
            // will be thrown. For this reason the previous response data is returned here and checked later.
            return previousRes;
        });

        if (protocols.isEmpty()) {
            redirectResponse.headers().firstValue("Location").ifPresent(url -> {
                Pattern pattern = Pattern.compile(".*/transfers/(.+)/results/transferDetails");
                Matcher matcher = pattern.matcher(url);
                if (matcher.matches()) {
                    String jobId = matcher.group(1);
                    String errorDetail = getErrorDetail(jobId, token);
                    if (!errorDetail.isBlank()) {
                        throw new BadRequestException(errorDetail);
                    }
                }
            });
            throw new BadRequestException("Protocol negotiation failed");
        }
        return protocols.get(0).getEndpoint();
    }

    public void createCollection(String name, Optional<String> token) {

        NodeCollection nc = new NodeCollection();
        nc.setTitle(name);

        HttpRequest request = getRequest("/collections", token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .PUT(HttpRequest.BodyPublishers.ofString(marshal(nc)))
                .build();

        call(request, BodyHandlers.ofInputStream(), 200, res -> null);
    }

    public void deleteCollection(Long id, Optional<String> token) {

        HttpRequest request = getRequest("/collections/"+id, token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .DELETE()
                .build();

        call(request, BodyHandlers.ofInputStream(), 200, res -> null);
    }

    public Node createNode(Node node, Optional<String> token) {

        String path = NodeUtils.getVosPath(node);

        HttpRequest request = getRequest("/nodes" + urlEncodePath(path), token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .PUT(HttpRequest.BodyPublishers.ofString(marshal(node)))
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> unmarshal(res, Node.class));
    }

    public void deleteNode(String path, Optional<String> token) {

        HttpRequest request = getRequest("/nodes" + urlEncodePath(path), token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .DELETE()
                .build();

        call(request, BodyHandlers.ofInputStream(), 200, res -> null);
    }

    public Node setNode(Node node, boolean recursive, Optional<String> token) {

        String path = NodeUtils.getVosPath(node);

        HttpRequest request = getRequest("/nodes" + urlEncodePath(path) + "?recursive=" + recursive, token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .POST(HttpRequest.BodyPublishers.ofString(marshal(node)))
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> unmarshal(res, Node.class));
    }

    public List<Job> getAsyncRecallJobs(Optional<String> token) {
        return getJobs("direction=pullToVoSpace", Job.JobType.ASYNC_RECALL, token);
    }

    public List<Job> getArchiveJobs(Optional<String> token) {
        return getJobs("direction=pullFromVoSpace"
                + "&VIEW=ivo://ia2.inaf.it/vospace/views%23tar"
                + "&VIEW=ivo://ia2.inaf.it/vospace/views%23zip",
                Job.JobType.ARCHIVE, token);
    }

    private List<Job> getJobs(String queryString, Job.JobType type, Optional<String> token) {

        HttpRequest request = getRequest("/transfers?" + queryString, token)
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .GET()
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> {
            return unmarshal(res, Jobs.class).getJobref().stream()
                    .map(jobDesc -> new Job(jobDesc, type))
                    .collect(Collectors.toList());
        });
    }

    public String getArchiveJobHref(String jobId, Optional<String> token) {
        List<Protocol> protocols = getTransferDetails(jobId, token).getProtocols();
        if (!protocols.isEmpty()) {
            return protocols.get(0).getEndpoint();
        }
        return null;
    }

    private Transfer getTransferDetails(String jobId, Optional<String> token) {

        HttpRequest request = getRequest("/transfers/" + jobId + "/results/transferDetails", token)
                .GET().build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> unmarshal(res, Transfer.class));
    }

    public ExecutionPhase getJobPhase(String jobId, Optional<String> token) {

        HttpRequest request = getRequest("/transfers/" + jobId + "/phase", token)
                .GET()
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> {
            try {
                return ExecutionPhase.valueOf(new String(res.readAllBytes()));
            } catch (IOException ex) {
                throw new UncheckedIOException(ex);
            }
        });
    }

    public String getErrorDetail(String jobId, Optional<String> token) {

        HttpRequest request = getRequest("/transfers/" + jobId + "/error", token)
                .header("Accept", "text/plain")
                .GET()
                .build();

        return call(request, BodyHandlers.ofString(), 200, res -> res);
    }

    private <T, U> U call(HttpRequest request, HttpResponse.BodyHandler<T> responseBodyHandler, int expectedStatusCode, Function<T, U> responseHandler) {
        return call(request, responseBodyHandler, expectedStatusCode, (res, prev) -> {
            return responseHandler.apply(res);
        });
    }

    private <T, U> U call(HttpRequest request, HttpResponse.BodyHandler<T> responseBodyHandler, int expectedStatusCode, BiFunction<T, HttpResponse<T>, U> responseHandler) {
        try {
            return httpClient.sendAsync(request, responseBodyHandler)
                    .thenApply(response -> {
                        if (response.statusCode() == expectedStatusCode) {
                            return response;
                        }
                        logServerError(request, response);
                        throw new VOSpaceStatusException("Error calling " + request.uri().toString() + ". Server response code is " + response.statusCode(), response.statusCode());
                    })
                    .thenApplyAsync(response -> {
                        HttpResponse<T> prev = response.previousResponse().orElse(null);
                        return responseHandler.apply(response.body(), prev);
                    }, jaxbExecutor)
                    .join();
        } catch (CompletionException ex) {
            if (ex.getCause() != null) {
                if (ex.getCause() instanceof ConnectException) {
                    throw new VOSpaceException("Cannot connect to " + request.uri().getHost() + " on port " + request.uri().getPort());
                }
                if (ex.getCause() instanceof VOSpaceException) {
                    throw (VOSpaceException) ex.getCause();
                }
                LOG.error("Error calling " + request.uri().toString(), ex.getCause());
                throw new VOSpaceException("Error calling " + request.uri().toString(), ex.getCause());
            }
            LOG.error("Error calling " + request.uri().toString(), ex);
            throw new VOSpaceException("Error calling " + request.uri().toString(), ex);
        }
    }

    private HttpRequest.Builder getRequest(String path, Optional<String> token) {
        HttpRequest.Builder builder = HttpRequest.newBuilder(URI.create(baseUrl + path));
        if (token.isPresent()) {
            builder.setHeader("Authorization", "Bearer " + token.get());
        }
        return builder;
    }

    private <T> T unmarshal(InputStream in, Class<T> type) {
        try {
            if (useJson) {
                return MAPPER.readValue(in, type);
            } else {
                return JAXB.unmarshal(in, type);
            }
        } catch (IOException ex) {
            LOG.error("Invalid JSON for class {}", type.getCanonicalName());
            throw new UncheckedIOException(ex);
        }
    }

    private String marshal(Object data) {
        try {
            if (useJson) {
                return MAPPER.writeValueAsString(data);
            } else {
                try ( StringWriter sw = new StringWriter()) {
                    JAXB.marshal(data, sw);
                    return sw.toString();
                }
            }
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private static <T> void logServerError(HttpRequest request, HttpResponse<T> response) {
        if (response.body() instanceof String) {
            logServerErrorString(request, (HttpResponse<String>) response);
        } else if (response.body() instanceof InputStream) {
            logServerErrorInputStream(request, (HttpResponse<InputStream>) response);
        } else {
            throw new UnsupportedOperationException("Unable to log error for response body type " + response.body().getClass().getCanonicalName());
        }
    }

    private static void logServerErrorString(HttpRequest request, HttpResponse<String> response) {

        LOG.error("Error while reading " + request.uri()
                + "\nServer response status code is " + response.statusCode()
                + "\nServer response text is " + response.body());
    }

    private static void logServerErrorInputStream(HttpRequest request, HttpResponse<InputStream> response) {
        Scanner s = new Scanner(response.body()).useDelimiter("\\A");
        String responseBody = s.hasNext() ? s.next() : "";
        String error = "Error while reading " + request.uri()
                + "\nServer response status code is " + response.statusCode()
                + "\nServer response text is " + responseBody;
        LOG.error(error);
    }
}
