/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.vospace.ui.TokenProvider;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.oats.vospace.datamodel.collections.NodeCollection;
import it.inaf.oats.vospace.datamodel.collections.NodeCollectionsWrapper;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class NodeCollectionsController extends BaseController {
    
    private static final Logger LOG = LoggerFactory.getLogger(NodeCollectionsController.class);
    
    @Autowired
    private VOSpaceClient client;
    
    @Autowired
    private TokenProvider tokenProvider;
    
    @GetMapping(value = "/collections", produces = MediaType.APPLICATION_JSON_VALUE)
    public List<NodeCollection> listNodeCollections(User principal) 
            throws Exception
    {
        LOG.debug("listNodeCollections called for user {}", principal.getName());
        
        NodeCollectionsWrapper ncw =
                client.getNodeCollections(tokenProvider.getToken());
        
        return ncw.getNodeCollections();
    }
    
    @PutMapping(value = "/collections", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<?> createNodeCollection(User principal, 
            @RequestBody Map<String, Object> params) 
            throws Exception
    {
        String collectionName = (String) this.getRequiredParam(params, "collectionName");
        
        LOG.debug("create new collection named {} for user {}", 
                collectionName, principal.getName());

        // Call creation logic        
        client.createCollection(collectionName, tokenProvider.getToken());
        
        return ResponseEntity.noContent().build();
    }
    
    @DeleteMapping(value = "/collections/{collectionId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<?> deleteNodeCollection(User principal, 
            @PathVariable Long collectionId) 
            throws Exception
    {                
        
        LOG.debug("delete collection {} for user {}", 
                collectionId, principal.getName());

        // Call creation logic        
        client.deleteCollection(collectionId, tokenProvider.getToken());
                
        return ResponseEntity.noContent().build();
    }
    
}
