/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.vospace.ui.TokenProvider;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.CreateLinkRequest;
import it.inaf.ia2.vospace.ui.exception.VOSpaceStatusException;
import java.util.Collections;
import net.ivoa.xml.vospace.v2.ContainerNode;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.multipart;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@AutoConfigureMockMvc
@TestPropertySource(properties = {"vospace-authority=example.com!vospace"})
public class CreateLinksControllerTest {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    @MockBean
    private VOSpaceClient client;

    @MockBean
    private TokenProvider tokenProvider;

    @Autowired
    private MockMvc mockMvc;

    @Mock
    private User user;

    @BeforeEach
    public void setUp() {
        when(user.getName()).thenReturn("user_id");
    }

    @Test
    public void testCreateSingleLink() throws Exception {

        ContainerNode myFolder = new ContainerNode();
        myFolder.setUri("vos://ia2.inaf.it!vospace/path/to/myfolder");
        when(client.getNode(eq("/path/to/myfolder"), any())).thenReturn(myFolder);

        CreateLinkRequest request = new CreateLinkRequest();

        request.setFolder("path/to/myfolder");
        request.setUrl("http://archives.ia2.inaf.it/files/aao/SC182172.fits.gz");
        request.setNodeName("myLink");

        String requestPayload = MAPPER.writeValueAsString(request);

        mockMvc.perform(post("/createLink")
                .contentType(MediaType.APPLICATION_JSON)
                .content(requestPayload)
                .sessionAttr("user_data", user))
                .andDo(print())
                .andExpect(status().isNoContent());

        verify(client, times(1)).createNode(argThat(node -> {
            return node.getUri().equals("vos://ia2.inaf.it!vospace/path/to/myfolder/myLink");
        }), any());
    }

    @Test
    public void testUploadLinksNonExistentFolder() throws Exception {

        when(client.getNode(eq("/path/to/non-existent"), any())).thenThrow(new VOSpaceStatusException("Not found", 404));

        mockMvc.perform(multipart("/uploadLinks")
                .file(getListOfLinksMockMultipartFile())
                .param("folder", "path/to/non-existent")
                .sessionAttr("user_data", user))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    public void testUploadLinks() throws Exception {

        ContainerNode myFolder = new ContainerNode();
        when(client.getNode(eq("/path/to/myfolder"), any())).thenReturn(myFolder);

        mockMvc.perform(multipart("/uploadLinks")
                .file(getListOfLinksMockMultipartFile())
                .param("folder", "path/to/myfolder")
                .sessionAttr("user_data", user))
                .andDo(print())
                .andExpect(status().isNoContent());

        verify(client, times(75)).createNode(any(), any());
    }

    @Test
    public void testInvalidUrl() throws Exception {
        testInvalidContent("foo");
    }

    @Test
    public void testInvalidUrlNoFile() throws Exception {
        testInvalidContent("http://archives.ia2.inaf.it/");
    }

    @Test
    public void testInvalidUrlNoFileNoSlash() throws Exception {
        testInvalidContent("http://archives.ia2.inaf.it");
    }

    @Test
    public void testTooManyLinks() throws Exception {
        testInvalidContent(String.join("\n", Collections.nCopies(1500, "http://archives.ia2.inaf.it/files/aao/SC182172.fits.gz")));
    }

    private void testInvalidContent(String fileContent) throws Exception {

        ContainerNode myFolder = new ContainerNode();
        when(client.getNode(eq("/path/to/myfolder"), any())).thenReturn(myFolder);

        MockMultipartFile file = new MockMultipartFile("file", fileContent.getBytes());

        mockMvc.perform(multipart("/uploadLinks")
                .file(file)
                .param("folder", "path/to/myfolder")
                .sessionAttr("user_data", user))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    private MockMultipartFile getListOfLinksMockMultipartFile() throws Exception {
        return new MockMultipartFile("file", UploadControllerTest.class.getClassLoader().getResourceAsStream("list-of-links.txt"));
    }
}
