/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui;

import it.inaf.ia2.aa.AuthConfig;
import it.inaf.ia2.aa.LoginFilter;
import it.inaf.ia2.aa.ServiceLocator;
import it.inaf.ia2.aa.ServletRapClient;
import it.inaf.ia2.aa.UserManager;
import it.inaf.ia2.gms.client.GmsClient;
import it.inaf.ia2.rap.client.ClientCredentialsRapClient;
import java.util.concurrent.ForkJoinPool;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.core.Ordered;
import org.springframework.web.client.RestTemplate;

@SpringBootApplication
public class VOSpaceUiApplication {

    public static void main(String[] args) {
        SpringApplication.run(VOSpaceUiApplication.class, args);
    }

    @Bean
    public FilterRegistrationBean preLoginFilterRegistration() {
        FilterRegistrationBean bean = new FilterRegistrationBean();
        bean.setFilter(new PreLoginFilter());
        bean.addUrlPatterns("/login");
        bean.setOrder(Ordered.HIGHEST_PRECEDENCE);
        return bean;
    }

    @Bean
    public FilterRegistrationBean loginFilterRegistration() {
        FilterRegistrationBean registration = new FilterRegistrationBean();
        registration.setFilter(new LoginFilter());
        registration.addUrlPatterns("/login");
        return registration;
    }

    @Bean
    public FilterRegistrationBean userFilterRegistration() {
        FilterRegistrationBean registration = new FilterRegistrationBean();
        registration.setFilter(new UserFilter());
        registration.addUrlPatterns("/*");
        return registration;
    }

    @Bean
    public UserManager userManager() {
        return ServiceLocator.getInstance().getUserManager();
    }

    /**
     * Solves a ClassLoader issue. See class JaxbForkJoinWorkerThreadFactory.
     */
    public static ForkJoinPool getJaxbExecutor() {
        JaxbForkJoinWorkerThreadFactory threadFactory = new JaxbForkJoinWorkerThreadFactory();
        int parallelism = Math.min(0x7fff /* copied from ForkJoinPool.java */, Runtime.getRuntime().availableProcessors());
        return new ForkJoinPool(parallelism, threadFactory, null, false);
    }

    @Bean
    public RestTemplate restTemplate() {
        return new RestTemplate();
    }

    @Bean
    public ClientCredentialsRapClient gmsRapClient() {

        AuthConfig config = ServiceLocator.getInstance().getConfig();

        ClientCredentialsRapClient rapClient = new ClientCredentialsRapClient(config.getRapBaseUri());
        rapClient.setClientId(config.getClientId());
        rapClient.setClientSecret(config.getClientSecret());
        return rapClient;
    }

    @Bean
    public GmsClient gmsClient() {
        String gmsBaseUri = ServiceLocator.getInstance().getConfig().getGmsUri();
        return new GmsClient(gmsBaseUri);
    }

    @Bean
    public ServletRapClient servletRapClient() {
        return (ServletRapClient) ServiceLocator.getInstance().getRapClient();
    }
}
