/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.aa.ServletRapClient;
import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.rap.client.call.TokenExchangeRequest;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.exception.PermissionDeniedException;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import static it.inaf.oats.vospace.datamodel.NodeUtils.urlEncodePath;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class DownloadController {

    private static final Logger LOG = LoggerFactory.getLogger(DownloadController.class);

    @Value("${vospace-authority}")
    private String authority;

    @Autowired
    private VOSpaceClient client;

    @Autowired
    private HttpServletRequest servletRequest;

    @Autowired
    private ServletRapClient rapClient;

    @GetMapping(value = "/download/**")
    public ResponseEntity<?> directDownload() {

        String path = getPath("/download/");
        LOG.debug("directDownload called for path {}", path);

        Transfer transfer = new Transfer();
        transfer.setDirection("pullFromVoSpace");
        transfer.setTarget("vos://" + authority + urlEncodePath(path));

        Protocol httpProtocol = new Protocol();
        httpProtocol.setUri("ivo://ivoa.net/vospace/core#httpget");
        transfer.getProtocols().add(httpProtocol);

        Protocol httpsProtocol = new Protocol();
        httpsProtocol.setUri("ivo://ivoa.net/vospace/core#httpsget");
        transfer.getProtocols().add(httpsProtocol);

        String url = client.getFileServiceEndpoint(transfer);
        HttpHeaders headers = new HttpHeaders();
        headers.set("Location", url);
        return new ResponseEntity<>(headers, HttpStatus.SEE_OTHER);
    }

    @GetMapping(value = "/download")
    public ResponseEntity<?> downloadJobResult(@RequestParam("jobId") String jobId) {

        LOG.debug("job result download called for jobId {}", jobId);

        String token = ((User) servletRequest.getUserPrincipal()).getAccessToken();

        if (token == null) {
            throw new PermissionDeniedException("Token is null");
        }

        String endpoint = client.getArchiveJobHref(jobId);

        TokenExchangeRequest exchangeRequest = new TokenExchangeRequest()
                .setSubjectToken(token)
                .setResource(endpoint);

        String newToken = rapClient.exchangeToken(exchangeRequest, servletRequest);

        String url = endpoint + "?token=" + newToken;
        HttpHeaders headers = new HttpHeaders();
        headers.set("Location", url);
        return new ResponseEntity<>(headers, HttpStatus.SEE_OTHER);
    }

    protected String getPath(String prefix) {
        String requestURL = servletRequest.getRequestURL().toString();
        return NodeUtils.getPathFromRequestURLString(requestURL, prefix);
    }
}
