/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.Job;
import it.inaf.ia2.vospace.ui.data.ListNodeData;
import it.inaf.ia2.vospace.ui.data.MoveOrCopyRequest;
import it.inaf.ia2.vospace.ui.exception.VOSpaceException;
import it.inaf.ia2.vospace.ui.service.MainNodesHtmlGenerator;
import it.inaf.ia2.vospace.ui.service.MoveOrCopyNodeModalHtmlGenerator;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import static it.inaf.oats.vospace.datamodel.NodeUtils.urlEncodePath;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Param;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import net.ivoa.xml.vospace.v2.View;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class NodesController extends BaseController {

    private static final Logger LOG = LoggerFactory.getLogger(NodesController.class);

    @Value("${vospace-authority}")
    private String authority;

    @Value("${pollingTimeout:15}")
    private int pollingTimeout;

    @Autowired
    private VOSpaceClient client;

    @Autowired
    private HttpServletRequest servletRequest;

    @GetMapping(value = {"/nodes", "/nodes/**"}, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<ListNodeData> listNodes(User principal) throws Exception {

        String path = getPath("/nodes/");
        LOG.debug("listNodes called for path {}", path);

        ListNodeData listNodeData = new ListNodeData();

        Node node = client.getNode(path);

        listNodeData.setWritable(NodeUtils.checkIfWritable(node, principal.getName(), principal.getGroups()));

        List<Node> linkedNodes = getLinkedNodes(node, false);

        MainNodesHtmlGenerator htmlGenerator = new MainNodesHtmlGenerator(node, principal, authority, linkedNodes);
        listNodeData.setHtml(htmlGenerator.generateNodes());

        return ResponseEntity.ok(listNodeData);
    }

    @GetMapping(value = "/nodesForMoveOrCopy", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<ListNodeData> listNodesForMoveOrCopyModal(@RequestParam("path") String path, @RequestParam("target") List<String> targetNodes, User principal) throws Exception {

        LOG.debug("nodesForMoveOrCopy called for path {}", path);

        ListNodeData listNodeData = new ListNodeData();

        Node node = client.getNode(path);

        listNodeData.setWritable(NodeUtils.checkIfWritable(node, principal.getName(), principal.getGroups()));

        List<Node> linkedNodes = getLinkedNodes(node, true);

        MoveOrCopyNodeModalHtmlGenerator htmlGenerator = new MoveOrCopyNodeModalHtmlGenerator(node, targetNodes, principal, authority, linkedNodes);
        listNodeData.setHtml(htmlGenerator.generateNodes());

        return ResponseEntity.ok(listNodeData);
    }

    private List<Node> getLinkedNodes(Node node, boolean onlyDirectories) throws Exception {

        List<Node> linkedNodes = new ArrayList<>();
        if (node instanceof ContainerNode) {
            ContainerNode container = (ContainerNode) node;
            List<LinkNode> linksToLoad = container.getNodes().stream().filter(n -> n instanceof LinkNode)
                    .map(n -> (LinkNode) n).collect(Collectors.toList());

            // it is necessary to load link nodes metadata to understand if the linked node represents
            // a container, a VOSpace file or an external file
            List<CompletableFuture<Node>> nodesCalls = new ArrayList<>();

            for (LinkNode linkNode : linksToLoad) {
                String prefix = "vos://" + authority;
                if (linkNode.getTarget().startsWith(prefix)) {
                    String linkPath = linkNode.getTarget().substring(prefix.length());
                    nodesCalls.add(CompletableFuture.supplyAsync(() -> client.getNode(linkPath), Runnable::run)
                            .exceptionally(ex -> null)); // null is returned in case of broken link
                } else {
                    linkedNodes.add(linkNode);
                }
            }

            CompletableFuture.allOf(nodesCalls.toArray(CompletableFuture[]::new)).join();

            for (CompletableFuture<Node> nodeCall : nodesCalls) {
                Node linkedNode = nodeCall.get();

                if (linkedNode != null) {
                    if (linkedNode instanceof ContainerNode) {
                        linkedNodes.add(linkedNode);
                    } else if (!onlyDirectories) {
                        linkedNodes.add(linkedNode);
                    }
                }
            }
        }

        return linkedNodes;
    }

    @PostMapping(value = "/folder")
    public void newFolder(@RequestBody Map<String, Object> params) {

        String parentPath = getRequiredParam(params, "parentPath");
        if (!parentPath.startsWith("/")) {
            parentPath = "/" + parentPath;
        }
        String name = getRequiredParam(params, "name");

        LOG.debug("newFolder called for path {}/{}", parentPath, name);

        ContainerNode node = new ContainerNode();
        node.setUri("vos://" + authority + urlEncodePath(parentPath + "/" + name));

        Property creator = new Property();
        creator.setUri("ivo://ivoa.net/vospace/core#creator");
        creator.setValue(getUser().getName());
        node.getProperties().add(creator);

        client.createNode(node);
    }

    @PostMapping(value = "/delete", consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<?> deleteNodes(@RequestBody List<String> nodesToDelete) {

        CompletableFuture[] deleteCalls = nodesToDelete.stream()
                .map(nodeToDelete -> {
                    return CompletableFuture.runAsync(() -> {
                        LOG.debug("deleteNode called for path {}", nodeToDelete);
                        client.deleteNode(nodeToDelete);
                    }, Runnable::run);
                })
                .collect(Collectors.toList())
                .toArray(CompletableFuture[]::new);

        Optional<RuntimeException> error = CompletableFuture.allOf(deleteCalls)
                .handle((fn, ex) -> {
                    return Optional.ofNullable(ex == null ? null : new RuntimeException(ex));
                }).join();

        if (error.isPresent()) {
            throw error.get();
        }

        // All the nodes have been correctly deleted
        return ResponseEntity.noContent().build();
    }

    @PostMapping(value = "/zip", consumes = MediaType.APPLICATION_JSON_VALUE)
    public Job createZip(@RequestBody List<String> paths) {
        return createArchive(paths, "ivo://ia2.inaf.it/vospace/views#zip");
    }

    @PostMapping(value = "/tar", consumes = MediaType.APPLICATION_JSON_VALUE)
    public Job createTar(@RequestBody List<String> paths) {
        return createArchive(paths, "ivo://ia2.inaf.it/vospace/views#tar");
    }

    private Job createArchive(List<String> paths, String viewUri) {

        Transfer transfer = new Transfer();

        View view = new View();
        view.setUri(viewUri);
        transfer.setView(view);

        if (paths.size() == 1) {
            transfer.setTarget("vos://" + authority + paths.get(0));
        } else {
            String parent = getCommonParent(paths);
            transfer.setTarget("vos://" + authority + parent);
            for (String path : paths) {
                String childName = path.substring(parent.length() + 1);
                Param param = new Param();
                param.setUri(viewUri + "/include");
                param.setValue(childName);
                view.getParam().add(param);
            }
        }

        transfer.setDirection("pullFromVoSpace");
        Protocol protocol = new Protocol();
        protocol.setUri("ivo://ivoa.net/vospace/core#httpget");
        transfer.getProtocols().add(protocol);

        return new Job(client.startTransferJob(transfer), Job.JobType.ARCHIVE);
    }

    private String getCommonParent(List<String> vosPaths) {
        String commonParent = null;
        for (String vosPath : vosPaths) {
            if (commonParent == null) {
                commonParent = vosPath;
            } else {
                StringBuilder newCommonParent = new StringBuilder();
                boolean same = true;
                int lastSlashPos = vosPath.lastIndexOf("/");
                for (int i = 0; same && i < Math.min(commonParent.length(), vosPath.length()) && i < lastSlashPos; i++) {
                    if (commonParent.charAt(i) == vosPath.charAt(i)) {
                        newCommonParent.append(commonParent.charAt(i));
                    } else {
                        same = false;
                    }
                }
                commonParent = newCommonParent.toString();
            }
        }
        return commonParent;
    }

    @PostMapping(value = "/moveOrCopy")
    public ResponseEntity<List<Job>> moveOrCopyNodes(@RequestBody MoveOrCopyRequest request) throws Exception {

        // Creates a transfer request for each copy or move operation
        CompletableFuture<JobSummary>[] futureJobs = request.getTargets().stream().map(t -> {
            String target = urlEncodePath(t);
            String direction = urlEncodePath(request.getDirection());
            Transfer transfer = new Transfer();
            transfer.setTarget("vos://" + authority + target);
            transfer.setDirection("vos://" + authority + direction);
            transfer.setKeepBytes(request.isKeepBytes());
            return CompletableFuture.supplyAsync(() -> client.startTransferJob(transfer), Runnable::run);
        }).collect(Collectors.toList()).toArray(CompletableFuture[]::new);

        // starts all HTTP requests in parallel
        CompletableFuture.allOf(futureJobs).join();

        // parses all the responses and populates a list of jobs to check the
        // completion status using polling mechanism
        List<JobSummary> jobs = Stream.of(futureJobs).map(j -> j.join()).collect(Collectors.toList());

        // Job statuses are checked for some time and if completion takes too much
        // time sends the execution phase to the UI and let it handles the polling.
        // Since CompletableFutures can't really be aborted (see cancel method
        // documentation) an AtomicReference containing a boolean flag it is
        // passed to handle polling abortion.
        AtomicReference<Boolean> cancelled = new AtomicReference<>(false);

        // CompletableFuture that triggers the timeout
        CompletableFuture timeout = CompletableFuture.runAsync(() -> {
            try {
                Thread.sleep(pollingTimeout * 1000);
            } catch (InterruptedException ex) {
            }
            cancelled.set(true);
        });

        try {
            // performs polling of job statuses or timeout
            CompletableFuture.anyOf(jobsPolling(jobs, cancelled), timeout).join();
        } catch (CompletionException ex) {
            if (ex.getCause() != null && ex.getCause() instanceof VOSpaceException) {
                throw (VOSpaceException) ex.getCause();
            }
            throw ex;
        }

        Job.JobType type = request.isKeepBytes() ? Job.JobType.COPY : Job.JobType.MOVE;

        return ResponseEntity.ok(jobs.stream().map(j -> new Job(j, type))
                .collect(Collectors.toList()));
    }

    private CompletableFuture<?> jobsPolling(List<JobSummary> jobs, AtomicReference<Boolean> cancelled) {
        return CompletableFuture.runAsync(() -> {

            List<JobSummary> uncompletedJobs;
            do {
                uncompletedJobs = jobs.stream()
                        .filter(j -> ExecutionPhase.COMPLETED != j.getPhase())
                        .collect(Collectors.toList());

                if (!uncompletedJobs.isEmpty()) {
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException ex) {
                        break;
                    }
                    updatePhases(uncompletedJobs);
                }
            } while (!uncompletedJobs.isEmpty() && !cancelled.get());
        }, Runnable::run);
    }

    private void updatePhases(List<JobSummary> uncompletedJobs) {
        CompletableFuture[] phaseFutures = uncompletedJobs.stream()
                .map(job -> CompletableFuture.runAsync(() -> {
            ExecutionPhase phase = client.getJobPhase(job.getJobId());
            if (phase == ExecutionPhase.ERROR) {
                String errorDetail = client.getErrorDetail(job.getJobId());
                throw new VOSpaceException("Operation failed: " + errorDetail);
            }
            job.setPhase(phase);
        }, Runnable::run)).collect(Collectors.toList()).toArray(CompletableFuture[]::new);

        CompletableFuture.allOf(phaseFutures).join();
    }

    protected String getPath(String prefix) {
        String requestURL = servletRequest.getRequestURL().toString();
        return NodeUtils.getPathFromRequestURLString(requestURL, prefix);
    }
}
