/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.PreUploadResult;
import it.inaf.ia2.vospace.ui.data.UploadFilesData;
import it.inaf.ia2.vospace.ui.exception.PermissionDeniedException;
import it.inaf.ia2.vospace.ui.exception.VOSpaceException;
import it.inaf.ia2.vospace.ui.exception.VOSpaceStatusException;
import static it.inaf.oats.vospace.datamodel.NodeUtils.urlEncodePath;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import javax.validation.Valid;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class UploadController extends BaseController {

    private static final Logger LOG = LoggerFactory.getLogger(UploadController.class);

    @Value("${vospace-authority}")
    private String authority;

    @Autowired
    private VOSpaceClient client;

    @PostMapping(value = "/preupload", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<List<PreUploadResult>> prepareForUpload(@RequestBody @Valid UploadFilesData data) {

        if (getUser() == null) {
            throw new PermissionDeniedException("File upload not allowed to anonymous users");
        }

        CompletableFuture<PreUploadResult>[] calls
                = data.getFiles().stream().map(fileName -> prepareForDownload(getParentPath(data), fileName))
                        .toArray(CompletableFuture[]::new);

        List<PreUploadResult> uploadUrls = CompletableFuture.allOf(calls)
                .thenApplyAsync(ignore -> {
                    return Arrays.stream(calls).map(c -> c.join()).collect(Collectors.toList());
                }).join();

        return ResponseEntity.ok(uploadUrls);
    }

    private String getParentPath(UploadFilesData data) {
        String parentPath = data.getParentPath();
        if (!parentPath.startsWith("/")) {
            parentPath = "/" + parentPath;
        }
        return parentPath;
    }

    public CompletableFuture<PreUploadResult> prepareForDownload(String parentPath, String fileName) {

        return CompletableFuture.supplyAsync(() -> {

            String path = parentPath;

            if (!path.endsWith("/")) {
                path += "/";
            }
            path += fileName;

            String nodeUri = "vos://" + authority + urlEncodePath(path);

            PreUploadResult result = new PreUploadResult();

            try {
                createDataNode(nodeUri, getUser().getName());
            } catch (Throwable t) {
                if (t instanceof VOSpaceStatusException && ((VOSpaceStatusException) t).getHttpStatus() == 409) {
                    result.setError("Node already exists");
                } else {
                    LOG.error("Error while creating node metadata for " + nodeUri, t);
                    result.setError("Unable to create node metadata");
                }
                return result;
            }

            try {
                String uploadUrl = obtainUploadUrl(nodeUri);
                result.setUrl(uploadUrl);
            } catch (Throwable t) {
                if (t instanceof VOSpaceException) {
                    result.setError(t.getMessage());
                } else {
                    LOG.error("Error while obtaining upload URL for " + nodeUri, t);
                    result.setError("Unable to obtain upload URL");
                }

                try {
                    // attempt to cleanup node metadata
                    client.deleteNode(path);
                } catch (Throwable dt) {
                    LOG.error("Unable to remove node after failed upload URL retrieval", dt);
                    result.setError("Retrieval of upload URL failed. Manual cleanup of node metadata may be necessary");
                }
            }
            return result;
        }, Runnable::run); // Passing current thread Executor to CompletableFuture to avoid "No thread-bound request found" exception
    }

    private void createDataNode(String nodeUri, String userId) {

        DataNode node = new DataNode();
        node.setUri(nodeUri);

        Property creator = new Property();
        creator.setUri("ivo://ivoa.net/vospace/core#creator");
        creator.setValue(userId);

        node.getProperties().add(creator);

        client.createNode(node);
    }

    private String obtainUploadUrl(String uri) {

        Transfer transfer = new Transfer();
        transfer.setDirection("pushToVoSpace");
        transfer.setTarget(uri);

        Protocol protocol = new Protocol();
        protocol.setUri("ivo://ivoa.net/vospace/core#httpput");
        transfer.getProtocols().add(protocol);

        return client.getFileServiceEndpoint(transfer);
    }
}
