/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.vospace.ui.exception.VOSpaceException;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Optional;
import java.util.stream.Collectors;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeInfo {

    private static final Logger LOG = LoggerFactory.getLogger(NodeInfo.class);

    private final String authority;

    private final String path;
    private final String name;
    private final String size;
    private String type;
    private final String creator;
    private final String groupRead;
    private final String groupWrite;
    private final boolean isPublic;
    private final boolean asyncTrans;
    private final boolean sticky;
    private final boolean busy;
    private final boolean writable;
    private final boolean deletable;
    private String target;

    public NodeInfo(Node node, User user, String authority, Node linkedNode) {
        this.authority = authority;
        this.path = getPath(node);
        this.name = URLDecoder.decode(path.substring(path.lastIndexOf("/") + 1).replace("+", "%2B"), StandardCharsets.UTF_8);
        this.size = getSize(node);
        this.type = node.getType();
        this.creator = getCreator(node);
        this.groupRead = getGroupRead(node);
        this.groupWrite = getGroupWrite(node);
        this.isPublic = isPublic(node);
        this.asyncTrans = isAsyncTrans(node);
        this.sticky = isSticky(node);
        this.busy = isBusy(node);
        this.writable = NodeUtils.checkIfWritable(node, user.getName(), user.getGroups()) && !busy;
        this.deletable = writable && !sticky && !asyncTrans;
        if (linkedNode != null) {
            String prefix = "vos://" + authority;
            this.target = decodePath(((LinkNode) node).getTarget(), prefix);
            this.type = linkedNode.getType();
        } else if (node instanceof LinkNode) {
            this.target = ((LinkNode) node).getTarget();
            this.type = "vos:DataNode"; // data link
        }
    }

    private String getPath(Node node) {

        String uri = node.getUri();

        String prefix = "vos://" + authority;

        if (!uri.startsWith(prefix)) {
            throw new VOSpaceException("Node authority is different from configured one! Configured is " + authority + ", but node URI is " + uri);
        }

        return decodePath(uri, prefix);
    }

    private String decodePath(String uri, String prefix) {

        return String.join("/", Arrays.stream(uri.substring(prefix.length()).split("/"))
                .map(p -> p.replace("+", "%2B"))
                .map(p -> URLDecoder.decode(p, StandardCharsets.UTF_8))
                .collect(Collectors.toList()));
    }

    private String getCreator(Node node) {
        return getProperty(node, NodeProperties.CREATOR_URI).orElse("");
    }

    private String getGroupRead(Node node) {
        return getProperty(node, NodeProperties.GROUP_READ_URI).orElse("");
    }

    private String getGroupWrite(Node node) {
        return getProperty(node, NodeProperties.GROUP_WRITE_URI).orElse("");
    }

    private boolean isPublic(Node node) {
        return getProperty(node, NodeProperties.PUBLIC_READ_URI).map(value -> "true".equals(value)).orElse(false);
    }

    private boolean isAsyncTrans(Node node) {
        return getProperty(node, NodeProperties.ASYNC_TRANS_URN).map(value -> "true".equals(value)).orElse(false);
    }

    private boolean isSticky(Node node) {
        return getProperty(node, NodeProperties.STICKY_URN).map(value -> "true".equals(value)).orElse(false);
    }

    private boolean isBusy(Node node) {
        return node instanceof DataNode && ((DataNode) node).isBusy();
    }

    private Optional<String> getProperty(Node node, String uri) {
        if (node.getProperties() != null && node.getProperties() != null) {
            for (Property property : node.getProperties()) {
                if (uri.equals(property.getUri())) {
                    return Optional.of(property.getValue());
                }
            }
        }
        return Optional.empty();
    }

    private String getSize(Node node) {
        return getProperty(node, "ivo://ivoa.net/vospace/core#length")
                .map(value -> {
                    try {
                        long bytes = Long.parseLong(value);
                        return getHumanReadableSize(bytes);
                    } catch (NumberFormatException ex) {
                        LOG.warn("Invalid length for node " + node.getUri() + ". Length is " + value);
                        return "";
                    }
                })
                .orElse("");
    }

    /**
     * Credits: https://stackoverflow.com/a/24805871/771431
     */
    private String getHumanReadableSize(long bytes) {
        if (bytes < 1024) {
            return bytes + " B";
        }
        int z = (63 - Long.numberOfLeadingZeros(bytes)) / 10;
        return String.format("%.1f %sB", (double) bytes / (1L << (z * 10)), " KMGTPE".charAt(z));
    }

    public boolean isFolder() {
        return "vos:ContainerNode".equals(type);
    }

    public boolean isFile() {
        return !"vos:ContainerNode".equals(type);
    }

    public String getPath() {
        return path;
    }

    public String getName() {
        return name;
    }

    public String getSize() {
        return size;
    }

    public String getCreator() {
        return creator;
    }

    public String getGroupRead() {
        return groupRead;
    }

    public String getGroupWrite() {
        return groupWrite;
    }

    public boolean isPublic() {
        return isPublic;
    }

    public boolean isAsyncTrans() {
        return asyncTrans;
    }

    public boolean isSticky() {
        return sticky;
    }

    public boolean isBusy() {
        return busy;
    }

    public boolean isWritable() {
        return writable;
    }

    public boolean isDeletable() {
        return deletable;
    }

    public String getTarget() {
        return target;
    }

    public boolean isLink() {
        return target != null;
    }
}
