/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.gms.client.GmsClient;
import it.inaf.ia2.gms.client.model.Permission;
import it.inaf.ia2.rap.client.ClientCredentialsRapClient;
import it.inaf.ia2.rap.data.Identity;
import it.inaf.ia2.rap.data.IdentityType;
import it.inaf.ia2.rap.data.RapUser;
import it.inaf.ia2.rap.data.TokenContext;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.ShareRequest;
import it.inaf.ia2.vospace.ui.data.SharingInfo;
import it.inaf.ia2.vospace.ui.exception.BadRequestException;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

@Service
public class SharingService {

    private static final Logger LOG = LoggerFactory.getLogger(SharingService.class);

    @Value("${trusted.eppn.scope}")
    protected String trustedEppnScope;

    private final Set<String> existingGroups = new CopyOnWriteArraySet<>();
    private final Set<String> existingPeopleGroups = new CopyOnWriteArraySet<>();
    private final Map<String, String> existingUsers = new ConcurrentHashMap<>();

    private final GmsClient gmsClient;
    private final ClientCredentialsRapClient rapClient;
    private final VOSpaceClient vospaceClient;

    private TokenContext tokenContext;
    private Date lastUpdate;

    @Autowired
    private HttpServletRequest request;

    @Autowired
    public SharingService(GmsClient gmsClient, ClientCredentialsRapClient gmsRapClient, VOSpaceClient vospaceClient) {
        this.gmsClient = gmsClient;
        this.rapClient = gmsRapClient;
        this.vospaceClient = vospaceClient;
    }

    public SharingInfo getSharingInfo() {

        loadInfoFromServices();

        SharingInfo sharingInfo = new SharingInfo();

        Set<String> people = new HashSet<>(existingUsers.values());
        people.addAll(existingPeopleGroups);

        sharingInfo.setPeople(new ArrayList<>(people));
        Collections.sort(sharingInfo.getPeople());

        sharingInfo.setGroups(new ArrayList<>(existingGroups));
        Collections.sort(sharingInfo.getGroups());

        return sharingInfo;
    }

    public void setNodeGroups(ShareRequest shareRequest) {

        loadInfoFromServices();

        createPeopleGroupIfNeeded(shareRequest.getUserRead());
        createPeopleGroupIfNeeded(shareRequest.getUserWrite());
        validateGroups(shareRequest.getGroupRead());
        validateGroups(shareRequest.getGroupWrite());

        String groupRead = getGroupString(shareRequest.getGroupRead(), shareRequest.getUserRead());
        String groupWrite = getGroupString(shareRequest.getGroupWrite(), shareRequest.getUserWrite());

        Node node = vospaceClient.getNode(shareRequest.getPath());
        getNodeProperty(node, NodeProperties.GROUP_READ_URI).setValue(groupRead);
        getNodeProperty(node, NodeProperties.GROUP_WRITE_URI).setValue(groupWrite);

        vospaceClient.setNode(node, shareRequest.isRecursive());
    }

    private Property getNodeProperty(Node node, String uri) {
        for (Property property : node.getProperties()) {
            if (uri.equals(property.getUri())) {
                return property;
            }
        }
        Property property = new Property();
        property.setUri(uri);
        node.getProperties().add(property);
        return property;
    }

    private void createPeopleGroupIfNeeded(List<String> users) {
        for (String username : users) {
            if (!existingPeopleGroups.contains(username)) {

                String completeGroupName = "people." + username.replace(".", "\\.");
                String rapId = getRapId(username);

                gmsClient.createGroup(completeGroupName, true);
                gmsClient.addMember(completeGroupName, rapId);
                gmsClient.addPermission(completeGroupName, rapId, Permission.VIEW_MEMBERS);

                existingPeopleGroups.add(username);
            }
        }
    }

    private String getRapId(String username) {
        for (Map.Entry<String, String> entry : existingUsers.entrySet()) {
            if (entry.getValue().equals(username)) {
                return entry.getKey();
            }
        }
        throw new BadRequestException("Unable to find an identifier for user " + username);
    }

    private void validateGroups(List<String> groups) {
        for (String group : groups) {
            if (!existingGroups.contains(group)) {
                throw new BadRequestException("Group " + group + " doesn't exist");
            }
        }
    }

    private String getGroupString(List<String> groups, List<String> users) {
        List<String> values = new ArrayList<>(groups);
        for (String user : users) {
            values.add("people." + user.replace(".", "\\."));
        }
        return String.join(" ", values);
    }

    private void loadInfoFromServices() {
        loadAccessToken();

        if (lastUpdate == null || new Date().getTime() - lastUpdate.getTime() > 3600 * 100) {
            LOG.trace("Loading existing users and groups from services");
            CompletableFuture.allOf(
                    CompletableFuture.runAsync(() -> loadGroups()),
                    CompletableFuture.runAsync(() -> loadUsers())
            ).join();
        }
    }

    public void loadAccessToken() {
        if (tokenContext == null) {
            tokenContext = new TokenContext();
        }
        if (tokenContext.isTokenExpired()) {
            tokenContext.setAccessTokenResponse(rapClient.getAccessTokenFromClientCredentials());
            gmsClient.setAccessToken(tokenContext.getAccessToken());
        }
    }

    private void loadGroups() {
        List<String> groups = new ArrayList<>();
        List<String> peopleGroups = new ArrayList<>();

        for (String group : gmsClient.listGroups("", true)) {
            if (group.startsWith("people.")) {
                String singleUserGroup = group.substring("people.".length()).replace("\\.", ".");
                peopleGroups.add(singleUserGroup);
            } else {
                groups.add(group);
            }
        }

        existingGroups.addAll(groups);
        existingPeopleGroups.addAll(peopleGroups);
    }

    private void loadUsers() {
        for (RapUser user : rapClient.getUsers(trustedEppnScope, tokenContext)) {
            for (Identity identity : user.getIdentities()) {
                if (identity.getType() == IdentityType.EDU_GAIN
                        && identity.getEppn().endsWith("@" + trustedEppnScope)) {
                    String username = identity.getEppn().substring(0, identity.getEppn().indexOf("@"));
                    existingUsers.put(user.getId(), username.toLowerCase());
                    break;
                }
            }
        }
    }
}
