/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
/* Vuex store, for centralized state management */

import Vue from 'vue';
import Vuex from 'vuex';
import client from 'api-client';
import main from './main';
import uploadsManager from './uploadsManager';

Vue.use(Vuex);

function updateArray(oldArr, newArr) {
  // empty the array
  oldArr.splice(0, oldArr.length);
  // fill again
  for (let i = 0; i < newArr.length; i++) {
    oldArr.push(newArr[i]);
  }
}

export default new Vuex.Store({
  state: {
    path: '',
    loading: true,
    nodesLoading: false,
    asyncButtonEnabled: false,
    deleteButtonEnabled: false,
    moveButtonEnabled: false,
    archiveButtonEnabled: false,
    copyButtonEnabled: false,
    jobs: [],
    jobsLoading: true,
    lastJobsCheckTime: null,
    user: 'anonymous',
    nodesToDelete: [],
    selectedUndeletableNodes: [],
    writable: false,
    nodeToShare: {
      path: null,
      groupRead: null,
      groupWrite: null
    },
    nodeToRename: null,
    moveOrCopy: 'move',
    nodesToMoveOrCopy: [],
    selectedNotMovableNodes: [],
    selectedNotCopiableNodes: [],
    moveOrCopyDestination: null,
    moveOrCopyDestinationWritable: false,
    archiveType: null,
    nodesToArchive: [],
    selectedNotArchivableNodes: []
  },
  modules: {
    uploadsManager
  },
  mutations: {
    setLoading(state, loading) {
      state.loading = loading;
    },
    setPath(state, value) {
      if (!value) {
        value = '';
      }
      state.path = value;
    },
    setNodesLoading(state, value) {
      state.nodesLoading = value;
    },
    setAsyncButtonEnabled(state, value) {
      state.asyncButtonEnabled = value;
    },
    setDeleteButtonEnabled(state, value) {
      state.deleteButtonEnabled = value;
    },
    setMoveButtonEnabled(state, value) {
      state.moveButtonEnabled = value;
    },
    setArchiveButtonEnabled(state, value) {
      state.archiveButtonEnabled = value;
    },
    setCopyButtonEnabled(state, value) {
      state.copyButtonEnabled = value;
    },
    setJobs(state, jobs) {
      updateArray(state.jobs, jobs);
    },
    addJob(state, job) {
      state.jobs.push(job);
    },
    setJobsLoading(state, loading) {
      state.jobsLoading = loading;
    },
    setUsername(state, username) {
      state.user = username;
    },
    setNodesToDelete(state, paths) {
      updateArray(state.nodesToDelete, paths);
    },
    setSelectedUndeletableNodes(state, paths) {
      updateArray(state.selectedUndeletableNodes, paths);
    },
    setSelectedNotCopiableNodes(state, paths) {
      updateArray(state.selectedNotCopiableNodes, paths);
    },
    setWritable(state, value) {
      state.writable = value;
    },
    setNodeToShare(state, data) {
      state.nodeToShare.path = data.path;
      state.nodeToShare.groupRead = data.groupRead;
      state.nodeToShare.groupWrite = data.groupWrite;
    },
    setNodeToRename(state, path) {
      state.nodeToRename = path;
    },
    setMoveOrCopy(state, value) {
      state.moveOrCopy = value;
    },
    setNodesToMoveOrCopy(state, paths) {
      updateArray(state.nodesToMoveOrCopy, paths);
    },
    setMoveOrCopyDestination(state, path) {
      state.moveOrCopyDestination = path;
    },
    setMoveOrCopyDestinationWritable(state, value) {
      state.moveOrCopyDestinationWritable = value;
    },
    setSelectedNotMovableNodes(state, paths) {
      updateArray(state.selectedNotMovableNodes, paths);
    },
    setArchiveType(state, type) {
      state.archiveType = type;
    },
    setNodesToArchives(state, paths) {
      state.nodesToArchive = paths;
    },
    setSelectedNotArchivableNodes(state, paths) {
      state.selectedNotArchivableNodes = paths;
    }
  },
  actions: {
    setPath({ state, commit, dispatch }, path) {
      commit('setPath', path);
      commit('setNodesLoading', true);
      client.getNode(state.path)
        .then(res => {
          dispatch('setNodes', res);
        })
        .finally(() => commit('setNodesLoading', false));
    },
    setNodes({ commit, dispatch }, res) {
      commit('setWritable', res.writable);
      document.getElementById('nodes').outerHTML = res.html;
      let checkboxes = document.querySelectorAll('#nodes input[type="checkbox"]');
      for (let i = 0; i < checkboxes.length; i++) {
        checkboxes[i].addEventListener('change', function() {
          dispatch('computeButtonsVisibility');
        });
      }
      dispatch('computeButtonsVisibility');
    },
    computeButtonsVisibility({ commit }) {
      commit('setAsyncButtonEnabled', document.querySelectorAll('#nodes input.async:checked').length > 0);
      commit('setDeleteButtonEnabled', document.querySelectorAll('#nodes input.deletable:checked').length > 0);
      commit('setMoveButtonEnabled', document.querySelectorAll('#nodes input.deletable:checked').length > 0);
      commit('setArchiveButtonEnabled', document.querySelectorAll('#nodes input:not(.async):checked').length > 0);
      commit('setCopyButtonEnabled', document.querySelectorAll('#nodes input:not(.async):checked').length > 0);
    },
    startAsyncRecallJob({ state, commit, dispatch }) {
      let asyncCheckboxes = document.querySelectorAll('#nodes input.async:checked');
      let paths = [];
      for (let i = 0; i < asyncCheckboxes.length; i++) {
        paths.push(asyncCheckboxes[i].getAttribute('data-node'));
      }
      client.startAsyncRecallJob(paths)
        .then(job => {
          main.showInfo('Job queued');
          commit('addJob', job);
          // Reload current node
          dispatch('setPath', state.path);
        });
    },
    checkJobs({ state, dispatch }) {
      if (state.jobs.filter(j => j.phase === 'QUEUED' || j.phase === 'PENDING' || j.phase === 'EXECUTING').length > 0 &&
        !state.jobsLoading && (state.lastJobsCheckTime !== null && new Date().getTime() - state.lastJobsCheckTime > 5000)) {
        dispatch('loadJobs');
      }
    },
    loadJobs({ state, commit }) {
      commit('setJobsLoading', true);
      client.loadJobs()
        .then(jobs => {
          for (let previousJob of state.jobs) {
            for (let newJob of jobs) {
              if (newJob.id === previousJob.id && newJob.phase !== previousJob.phase) {
                switch (newJob.phase) {
                  case 'EXECUTING':
                    main.showInfo('Job started');
                    break;
                  case 'COMPLETED':
                    main.showInfo('Job completed');
                    break;
                  case 'ERROR':
                    main.showError('Job failed');
                    break;
                }
              }
            }
          }
          commit('setJobs', jobs);
        })
        .finally(() => {
          commit('setJobsLoading', false);
          state.lastJobsCheckTime = new Date().getTime();
        });
    },
    loadUserInfo({ commit }) {
      client.getUserInfo()
        .then(res => commit('setUsername', res.username));
    },
    createFolder({ state, dispatch }, newFolderName) {
      client.createFolder(state.path, newFolderName)
        .then(() => {
          // Reload current node
          dispatch('setPath', state.path);
        });
    },
    deleteNodes({ state, dispatch }) {
      client.deleteNodes(state.nodesToDelete)
        .then(() => {
          // Reload current node
          dispatch('setPath', state.path);
        });
    },
    moveOrCopyNodes({ state, commit, dispatch }, data) {
      client.moveOrCopyNodes(data)
        .then(jobs => {
          let uncompletedJobs = jobs.filter(j => j.phase !== 'COMPLETED');
          if (uncompletedJobs.length === 0) {
            // Reload current node
            dispatch('setPath', state.path);
          } else {
            main.showInfo('The operation is taking some time and will be handled in background');
            for (let job of uncompletedJobs) {
              commit('addJob', job);
            }
          }
        });
    },
    openNodeInMoveOrCopyModal({ state, commit }, path) {
      commit('setMoveOrCopyDestination', path);
      client.getNodesForMoveOrCopy({ path, targets: state.nodesToMoveOrCopy })
        .then(res => {
          commit('setMoveOrCopyDestinationWritable', res.writable);
          document.getElementById('move-or-copy-nodes').outerHTML = res.html;
        });
    },
    createArchive({ state, commit }) {
      client.createArchive(state.nodesToArchive, state.archiveType)
        .then(job => {
          if (job.phase === 'ERROR') {
            main.showError('Error creating ' + state.archiveType + ' archive');
          } else {
            main.showInfo(state.archiveType + ' creation started');
            commit('addJob', job);
          }
        });
    }
  }
});
