package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import java.io.IOException;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.Node;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

@Service
public class NodesService {

    private static final Logger LOG = LoggerFactory.getLogger(NodesService.class);

    @Autowired
    private VOSpaceClient client;

    @Value("${vospace-authority}")
    private String authority;

    public String generateNodesHtml(String path) {

        Node node = client.getNode(path);

        try ( StringWriter sw = new StringWriter()) {

            if (node instanceof ContainerNode) {
                ContainerNode folder = (ContainerNode) node;
                sw.write("<tbody id=\"nodes\">");
                for (Node child : folder.getNodes()) {
                    sw.write(getNodeHtml(child));
                }
                sw.write("</tbody>");
            }

            return sw.toString();
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private String getNodeHtml(Node node) {

        NodeInfo nodeInfo = new NodeInfo(node, authority);

        if (nodeInfo.getName().startsWith(".")) {
            // hidden file
            return "";
        }

        String html = "<tr>";
        html += "<td><input type=\"checkbox\" data-node=\"" + nodeInfo.getPath() + "\" ";
        if (nodeInfo.isAsyncTrans()) {
            html += "class=\"async\"";
        }
        html += "/></td>";
        html += "<td>" + getIcon(nodeInfo) + getLink(nodeInfo) + "</td>";
        html += "<td>" + nodeInfo.getSize() + "</td>";
        html += "<td>" + nodeInfo.getGroupRead() + "</td>";
        html += "<td>" + nodeInfo.getGroupWrite() + "</td>";
        html += "<td><span class=\"icon trash-icon pointer\" onclick=\"deleteNode('" + nodeInfo.getPath() + "')\"></span></td>";
        html += "</tr>";
        return html;
    }

    private String getIcon(NodeInfo nodeInfo) {
        String html = "<span class=\"icon ";
        if (nodeInfo.isFolder()) {
            html += "folder";
        } else {
            html += "file";
        }
        if (nodeInfo.isAsyncTrans()) {
            html += "-x";
        }
        html += "-icon\"></span>&nbsp;";
        return html;
    }

    private String getLink(NodeInfo nodeInfo) {
        if (isDownloadable(nodeInfo)) {
            if (nodeInfo.isFolder()) {
                return "<a href=\"#/nodes" + nodeInfo.getPath() + "\">" + nodeInfo.getName() + "</a>";
            } else {
                return "<a href=\"download" + nodeInfo.getPath() + "\" target=\"blank_\">" + nodeInfo.getName() + "</a>";
            }
        }
        return nodeInfo.getName();
    }

    private boolean isDownloadable(NodeInfo nodeInfo) {
        if (nodeInfo.isFile() && nodeInfo.isAsyncTrans()) {
            return false;
        }
        if (nodeInfo.isPublic()) {
            return true;
        }
        // TODO: check user group
        return true; // temporary always true
    }
}
