/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui;

import java.util.concurrent.ThreadFactory;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

/**
 * VOSpaceClient needs to extract the access token stored into the HTTP session
 * and forward it to the VOSpace REST service. The client can retrieve the
 * session from the autowired HttpServletRequest, however this doesn't work if
 * the client is called from a standard new thread (for example when the client
 * is invoked from a CompletableFuture), because Spring Context is missing and
 * retrieval of the autowired HttpServletRequest proxy fails with the "No
 * thread-bound request found" exception. Many CompletableFuture methods accept
 * an Executor as parameter, so this ThreadFactory has been created in order to
 * store the session in these particular threads. An Executor using this
 * ThreadaFactory must be passed to the CompletableFuture. An alternative would
 * be passing the original HttpServletRequest to all VOSpaceClient methods. A
 * first attempt using the current executor (Runnable::run) shown that tasks
 * where executed sequentially, so this dedicated Executor has been set up.
 */
public class SessionThreadFactory implements ThreadFactory {

    private final HttpServletRequest request;

    public SessionThreadFactory(HttpServletRequest request) {
        this.request = request;
    }

    @Override
    public Thread newThread(Runnable runnable) {
        return new SessionThread(runnable, request);
    }

    public static class SessionThread extends Thread {

        private final HttpSession session;

        public SessionThread(Runnable runnable, HttpServletRequest request) {
            super(runnable);
            this.session = request.getSession(false);
        }

        public HttpSession getHttpSession() {
            return this.session;
        }
    }
}
