/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.aa.data.User;
import java.util.List;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import net.ivoa.xml.vospace.v2.Node;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;

public abstract class NodesHtmlGenerator {

    private final Node parentNode;

    protected final User user;
    protected final String authority;
    private final List<Node> linkedNodes;

    private Element containerElement;

    protected NodesHtmlGenerator(Node node, User user, String authority, List<Node> linkedNodes) {
        this.parentNode = node;
        this.user = user;
        this.authority = authority;
        this.linkedNodes = linkedNodes;
    }

    public String generateNodes() {

        Document html = Jsoup.parse("<html></html>");

        containerElement = createContainerElement(html);

        if (parentNode instanceof ContainerNode) {
            ContainerNode folder = (ContainerNode) parentNode;
            for (Node child : folder.getNodes()) {
                addChild(child, containerElement);
            }
        }

        return containerElement.toString();
    }

    protected abstract Element createContainerElement(Document doc);

    protected abstract void addChild(Node child, Element containerElement);

    protected String makeJsArg(String arg) {
        return "'" + arg.replace("\\", "\\\\").replace("'", "\\'") + "'";
    }

    protected void addNodeIcon(NodeInfo nodeInfo, Element parentElement) {

        if (nodeInfo.isBusy()) {
            Element loadingWrapper = parentElement.appendElement("span");
            loadingWrapper.addClass("node-busy");
            Element spinner = loadingWrapper.appendElement("span");
            spinner.attr("role", "status");
            spinner.addClass("spinner-border");
            Element srEl = spinner.appendElement("span");
            srEl.addClass("sr-only");
            srEl.text("Loading...");
        }

        Element icon = parentElement.appendElement("span");
        icon.addClass("icon");

        if (nodeInfo.isLink()) {
            if (nodeInfo.isFolder()) {
                icon.addClass("folder-link-icon");
            } else if(nodeInfo.isBrokenLink()) {
                icon.addClass("question-icon");
            } else {
                icon.addClass("link-icon");
            }
        } else if (nodeInfo.isFolder()) {
            if (nodeInfo.isAsyncTrans()) {
                icon.addClass("folder-x-icon");
            } else {
                icon.addClass("folder-icon");
            }
        } else {
            if (nodeInfo.isAsyncTrans()) {
                icon.addClass("file-x-icon");
            } else {
                icon.addClass("file-icon");
            }
        }

        parentElement.append("&nbsp;");
    }

    protected Node getLinkedNode(Node node) {
        if (node instanceof LinkNode) {
            return linkedNodes.stream().filter(n -> ((LinkNode) node)
                    .getTarget().equals(n.getUri())).findFirst().orElse(null);
        }
        return null;
    }
}
