/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.client;

import it.inaf.ia2.vospace.ui.TokenProvider;
import it.inaf.ia2.vospace.ui.exception.BadRequestException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpHeaders;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.test.util.ReflectionTestUtils;

@ExtendWith(MockitoExtension.class)
public class VOSpaceClientTest {

    private HttpClient mockedHttpClient;
    private VOSpaceClient voSpaceClient;

    @BeforeEach
    public void init() {
        mockedHttpClient = mock(HttpClient.class);

        HttpClient.Builder builder = mock(HttpClient.Builder.class);
        when(builder.followRedirects(any())).thenReturn(builder);
        when(builder.version(any())).thenReturn(builder);
        when(builder.build()).thenReturn(mockedHttpClient);

        try ( MockedStatic<HttpClient> staticMock = Mockito.mockStatic(HttpClient.class)) {
            staticMock.when(HttpClient::newBuilder).thenReturn(builder);
            voSpaceClient = new VOSpaceClient("http://localhost/vospace");
        }
    }

    @Test
    public void testGetXmlNode() {
        ReflectionTestUtils.setField(voSpaceClient, "useJson", false);

        CompletableFuture response = getMockedStreamResponseFuture(200, getResourceFileContent("nodes-response.xml"));
        when(mockedHttpClient.sendAsync(any(), any())).thenReturn(response);

        ContainerNode node = (ContainerNode) voSpaceClient.getNode("/node1", Optional.empty());
        assertEquals("vos://ia2.inaf.it!vospace/node1", node.getUri());
    }

    @Test
    public void testCreateNode() {

        ReflectionTestUtils.setField(voSpaceClient, "useJson", false);

        CompletableFuture response = getMockedStreamResponseFuture(200, getResourceFileContent("node-response.xml"));
        when(mockedHttpClient.sendAsync(argThat(request -> {
            assertEquals("/vospace/nodes/mynode/newnode", request.uri().getPath());
            return true;
        }), any())).thenReturn(response);

        ContainerNode newNode = new ContainerNode();
        newNode.setUri("vos://ia2.inaf.it!vospace/mynode/newnode");

        ContainerNode responseNode = (ContainerNode) voSpaceClient.createNode(newNode, Optional.empty());
        assertEquals(newNode.getUri(), responseNode.getUri());
    }

    @Test
    public void testCreateNodeBadUri() {

        ContainerNode newNode = new ContainerNode();
        newNode.setUri("vos://ia2.inaf.it!vospace/mynode/spaces not encoded");

        try {
            voSpaceClient.createNode(newNode, Optional.empty());
            fail("Exception was expected");
        } catch (RuntimeException ex) {
            assertTrue(ex.getCause() instanceof URISyntaxException);
        }
    }

    @Test
    public void testGetErrorDetail() {

        CompletableFuture response = getMockedStringResponseFuture(200, "error message");
        when(mockedHttpClient.sendAsync(any(), any())).thenReturn(response);

        assertEquals("error message", voSpaceClient.getErrorDetail("123", Optional.empty()));
    }

    @Test
    public void testGetFileServiceEndpointSuccess() throws Exception {

        Transfer transfer = new Transfer();
        transfer.setDirection("pushToVoSpace");
        transfer.setTarget("vos://ia2.inaf.it!vospace/mynode");

        Protocol protocol = new Protocol();
        protocol.setUri("ivo://ivoa.net/vospace/core#httpput");
        transfer.getProtocols().add(protocol);

        CompletableFuture response = getMockedStreamResponseFuture(200, getResourceFileContent("transfer-response-ok.xml"));
        when(mockedHttpClient.sendAsync(any(), any())).thenReturn(response);

        assertEquals("http://storage1.example.com/trans/mynode", voSpaceClient.getFileServiceEndpoint(transfer, Optional.empty()));
    }

    @Test
    public void testGetFileServiceEndpointError() {

        Transfer transfer = new Transfer();
        transfer.setDirection("pushToVoSpace");
        transfer.setTarget("vos://ia2.inaf.it!vospace/mynode");

        Protocol protocol = new Protocol();
        protocol.setUri("ivo://ivoa.net/vospace/core#httpput");
        transfer.getProtocols().add(protocol);

        HttpResponse<InputStream> redirectResponse = mock(HttpResponse.class);
        HttpHeaders headers = mock(HttpHeaders.class);
        when(headers.firstValue("Location")).thenReturn(Optional.of("/vospace/transfers/1234/results/transferDetails"));
        when(redirectResponse.headers()).thenReturn(headers);

        HttpResponse<InputStream> mockedStreamResponse = getMockedStreamResponse(200, getResourceFileContent("transfer-response-no-protocols.xml"));
        when(mockedStreamResponse.previousResponse()).thenReturn(Optional.of(redirectResponse));

        CompletableFuture response1 = CompletableFuture.completedFuture(mockedStreamResponse);
        CompletableFuture response2 = getMockedStringResponseFuture(200, "error message");
        when(mockedHttpClient.sendAsync(any(), any())).thenReturn(response1).thenReturn(response2);

        BadRequestException ex = assertThrows(BadRequestException.class, () -> {
            voSpaceClient.getFileServiceEndpoint(transfer, Optional.empty());
        });
        assertEquals("error message", ex.getMessage());
    }

    @Test
    public void testGetFileServiceEndpointErrorWithoutMessage() {

        Transfer transfer = new Transfer();
        transfer.setDirection("pushToVoSpace");
        transfer.setTarget("vos://ia2.inaf.it!vospace/mynode");

        Protocol protocol = new Protocol();
        protocol.setUri("ivo://ivoa.net/vospace/core#httpput");
        transfer.getProtocols().add(protocol);

        HttpResponse<InputStream> redirectResponse = mock(HttpResponse.class);
        HttpHeaders headers = mock(HttpHeaders.class);
        when(headers.firstValue("Location")).thenReturn(Optional.of("/vospace/transfers/1234/results/transferDetails"));
        when(redirectResponse.headers()).thenReturn(headers);

        HttpResponse<InputStream> mockedStreamResponse = getMockedStreamResponse(200, getResourceFileContent("transfer-response-no-protocols.xml"));
        when(mockedStreamResponse.previousResponse()).thenReturn(Optional.of(redirectResponse));

        CompletableFuture response1 = CompletableFuture.completedFuture(mockedStreamResponse);
        CompletableFuture response2 = getMockedStringResponseFuture(200, "");
        when(mockedHttpClient.sendAsync(any(), any())).thenReturn(response1).thenReturn(response2);

        BadRequestException ex = assertThrows(BadRequestException.class, () -> {
            voSpaceClient.getFileServiceEndpoint(transfer, Optional.empty());
        });
        assertEquals("Protocol negotiation failed", ex.getMessage());
    }

    @Test
    public void testGetJobPhase() {

        CompletableFuture response = getMockedStreamResponseFuture(200, "COMPLETED");
        when(mockedHttpClient.sendAsync(any(), any())).thenReturn(response);

        assertEquals(ExecutionPhase.COMPLETED, voSpaceClient.getJobPhase("job_id", Optional.empty()));
    }

    @Test
    public void testSetNode() {

        ContainerNode node = new ContainerNode();
        node.setUri("vos://ia2.inaf.it!vospace/my+node");

        CompletableFuture response = getMockedStreamResponseFuture(200, getResourceFileContent("node-response.xml"));
        when(mockedHttpClient.sendAsync(any(), any())).thenReturn(response);

        voSpaceClient.setNode(node, true, Optional.empty());

        // verifying proper URL encoding of + char
        verify(mockedHttpClient).sendAsync(argThat(req -> {
            return req.uri().toString().contains("/nodes/my%2Bnode");
        }), any());
    }

    @Test
    public void testDeleteNode() {

        CompletableFuture response = getMockedStreamResponseFuture(200, "");
        when(mockedHttpClient.sendAsync(any(), any())).thenReturn(response);

        voSpaceClient.deleteNode("/not urlencoded", Optional.of("<token>"));
    }

    protected static String getResourceFileContent(String fileName) {
        try ( InputStream in = VOSpaceClientTest.class.getClassLoader().getResourceAsStream(fileName)) {
            return new String(in.readAllBytes(), StandardCharsets.UTF_8);
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    protected static CompletableFuture<HttpResponse<InputStream>> getMockedStreamResponseFuture(int statusCode, String body) {
        return CompletableFuture.completedFuture(getMockedStreamResponse(200, body));
    }

    protected static CompletableFuture<HttpResponse<String>> getMockedStringResponseFuture(int statusCode, String body) {
        return CompletableFuture.completedFuture(getMockedStringResponse(200, body));
    }

    protected static HttpResponse<InputStream> getMockedStreamResponse(int statusCode, String body) {
        HttpResponse response = getMockedResponse(statusCode);
        InputStream in = new ByteArrayInputStream(body.getBytes(StandardCharsets.UTF_8));
        when(response.body()).thenReturn(in);
        return response;
    }

    protected static HttpResponse<String> getMockedStringResponse(int statusCode, String body) {
        HttpResponse response = getMockedResponse(statusCode);
        when(response.body()).thenReturn(body);
        return response;
    }

    protected static HttpResponse getMockedResponse(int statusCode) {
        HttpResponse response = mock(HttpResponse.class);
        when(response.statusCode()).thenReturn(statusCode);
        return response;
    }
}
