package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.aa.data.User;
import static it.inaf.oats.vospace.datamodel.NodeUtils.urlEncodePath;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.Node;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;

public class NodesHtmlGenerator {

    private final Node parentNode;
    private final User user;
    private final String authority;

    private Element tbody;

    public NodesHtmlGenerator(Node node, User user, String authority) {
        this.parentNode = node;
        this.user = user;
        this.authority = authority;
    }

    public String generateNodes() {

        Document html = Jsoup.parse("<html></html>");

        tbody = html.body().appendElement("tbody");
        tbody.attr("id", "nodes");

        if (parentNode instanceof ContainerNode) {
            ContainerNode folder = (ContainerNode) parentNode;
            for (Node child : folder.getNodes()) {
                addChild(child);
            }
        }

        return tbody.toString();
    }

    private void addChild(Node child) {
        NodeInfo nodeInfo = new NodeInfo(child, user, authority);

        if (nodeInfo.isListOfFiles()) {
            // hidden file
            return;
        }

        Element row = tbody.appendElement("tr");

        addSelectionCell(nodeInfo, row);
        addLinkCell(nodeInfo, row);
        addSizeCell(nodeInfo, row);
        addGroupReadCell(nodeInfo, row);
        addGroupWriteCell(nodeInfo, row);
        addActionsCell(nodeInfo, row);
    }

    private void addSelectionCell(NodeInfo nodeInfo, Element row) {

        Element cell = row.appendElement("td");

        Element input = cell.appendElement("input");
        input.attr("type", "checkbox");
        input.attr("data-node", nodeInfo.getPath());

        if (nodeInfo.isAsyncTrans()) {
            input.addClass("async");
        } else if (nodeInfo.isDeletable()) {
            input.addClass("deletable");
        }
    }

    private void addLinkCell(NodeInfo nodeInfo, Element row) {

        Element cell = row.appendElement("td");
        addNodeIcon(nodeInfo, cell);
        addLink(nodeInfo, cell);
    }

    private void addSizeCell(NodeInfo nodeInfo, Element row) {
        Element cell = row.appendElement("td");
        cell.text(nodeInfo.getSize());
    }

    private void addGroupReadCell(NodeInfo nodeInfo, Element row) {
        Element cell = row.appendElement("td");
        fillGroupCell(cell, nodeInfo.getGroupRead());
    }

    private void addGroupWriteCell(NodeInfo nodeInfo, Element row) {
        Element cell = row.appendElement("td");
        fillGroupCell(cell, nodeInfo.getGroupWrite());
    }

    private void fillGroupCell(Element cell, String groups) {
        String[] values = groups.split(" ");
        List<String> personGroups = new ArrayList<>();
        List<String> peopleGroups = new ArrayList<>();
        for (String value : values) {
            if (value.startsWith("people.")) {
                personGroups.add(value.substring("people.".length()).replace("\\.", "."));
            } else {
                peopleGroups.add(value);
            }
        }
        if (!personGroups.isEmpty()) {
            Element personIcon = cell.appendElement("span");
            personIcon.attr("class", "icon person-icon");
            cell.appendText(String.join(" ", personGroups));
            cell.append("&nbsp;");
        }
        if (!peopleGroups.isEmpty()) {
            Element personIcon = cell.appendElement("span");
            personIcon.attr("class", "icon people-icon");
            cell.appendText(String.join(" ", peopleGroups));
        }
    }

    private void addActionsCell(NodeInfo nodeInfo, Element row) {
        Element cell = row.appendElement("td");
        if (nodeInfo.isWritable()) {
            Element shareIcon = cell.appendElement("span");
            shareIcon.attr("class", "icon share-icon pointer");
            shareIcon.attr("onclick", "shareNode(" + makeJsArg(nodeInfo.getPath())
                    + "," + makeJsArg(nodeInfo.getGroupRead())
                    + "," + makeJsArg(nodeInfo.getGroupWrite()) + ")");
        }
        if (nodeInfo.isDeletable()) {
            cell.append("&nbsp;");
            Element deleteIcon = cell.appendElement("span");
            deleteIcon.attr("class", "icon trash-icon pointer");
            deleteIcon.attr("onclick", "deleteNode(" + makeJsArg(nodeInfo.getPath()) + ")");
        }
    }

    private String makeJsArg(String arg) {
        return "'" + arg.replace("\\", "\\\\").replace("'", "\\'") + "'";
    }

    private void addNodeIcon(NodeInfo nodeInfo, Element cell) {

        Element iconContainer = cell;

        if (nodeInfo.isBusy()) {
            Element loadingWrapper = cell.appendElement("span");
            iconContainer = loadingWrapper;
            loadingWrapper.addClass("node-busy");
            Element spinner = loadingWrapper.appendElement("span");
            spinner.attr("role", "status");
            spinner.addClass("spinner-border");
            Element srEl = spinner.appendElement("span");
            srEl.addClass("sr-only");
            srEl.text("Loading...");
        }

        Element icon = cell.appendElement("span");
        icon.addClass("icon");

        if (nodeInfo.isFolder()) {
            if (nodeInfo.isAsyncTrans()) {
                icon.addClass("folder-x-icon");
            } else {
                icon.addClass("folder-icon");
            }
        } else {
            if (nodeInfo.isAsyncTrans()) {
                icon.addClass("file-x-icon");
            } else {
                icon.addClass("file-icon");
            }
        }

        iconContainer.append("&nbsp;");
    }

    private void addLink(NodeInfo nodeInfo, Element cell) {
        if (isDownloadable(nodeInfo, user)) {
            Element link = cell.appendElement("a");
            String href;
            if (nodeInfo.isFolder()) {
                href = "#/nodes" + urlEncodePath(nodeInfo.getPath());
            } else {
                href = "download" + urlEncodePath(nodeInfo.getPath());
                link.attr("target", "blank_");
            }
            link.attr("href", href);
            link.text(nodeInfo.getName());
        } else {
            cell.text(nodeInfo.getName());
        }
    }

    private boolean isDownloadable(NodeInfo nodeInfo, User user) {
        if (nodeInfo.isFile()) {
            if (nodeInfo.isAsyncTrans() || nodeInfo.isBusy()) {
                return false;
            }
        }
        if (nodeInfo.isPublic()) {
            return true;
        }

        if (nodeInfo.getCreator().equals(user.getName())) {
            return true;
        }

        if (user.getGroups() != null && !user.getGroups().isEmpty() && !nodeInfo.getGroupRead().isEmpty()) {
            List<String> groupRead = Arrays.asList(nodeInfo.getGroupRead().split(" "));
            for (String group : groupRead) {
                if (user.getGroups().contains(group)) {
                    return true;
                }
            }
        }

        return false;
    }
}
