package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.vospace.ui.exception.VOSpaceException;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import java.util.List;
import java.util.Optional;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.View;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeInfo {

    private static final Logger LOG = LoggerFactory.getLogger(NodeInfo.class);

    private final String authority;

    private final String path;
    private final String name;
    private final String size;
    private final String type;
    private final String creator;
    private final String groupRead;
    private final String groupWrite;
    private final boolean isPublic;
    private final boolean asyncTrans;
    private final boolean busy;
    private final boolean listOfFiles;

    public NodeInfo(Node node, String authority) {
        this.authority = authority;
        this.path = getPath(node);
        this.name = path.substring(path.lastIndexOf("/") + 1);
        this.size = getSize(node);
        this.type = node.getType();
        this.creator = getCreator(node);
        this.groupRead = getGroupRead(node);
        this.groupWrite = getGroupWrite(node);
        this.isPublic = isPublic(node);
        this.asyncTrans = isAsyncTrans(node);
        this.busy = isBusy(node);
        this.listOfFiles = isListOfFiles(node);
    }

    private String getPath(Node node) {

        String uri = node.getUri();

        String prefix = "vos://" + authority;

        if (!uri.startsWith(prefix)) {
            throw new VOSpaceException("Node authority is different from configured one! Configured is " + authority + ", but node URI is " + uri);
        }

        return uri.substring(prefix.length());
    }
    
    private String getCreator(Node node) {
        return getProperty(node, NodeProperties.CREATOR_URI).orElse("");
    }

    private String getGroupRead(Node node) {
        return getProperty(node, NodeProperties.GROUP_READ_URI).orElse("");
    }

    private String getGroupWrite(Node node) {
        return getProperty(node, NodeProperties.GROUP_WRITE_URI).orElse("");
    }

    private boolean isPublic(Node node) {
        return getProperty(node, NodeProperties.PUBLIC_READ_URI).map(value -> "t".equals(value)).orElse(false);
    }

    private boolean isAsyncTrans(Node node) {
        return getProperty(node, "urn:async_trans").map(value -> "t".equals(value)).orElse(false);
    }
    
    private boolean isBusy(Node node) {
        return node instanceof DataNode && ((DataNode) node).isBusy();
    }

    private boolean isListOfFiles(Node node) {
        if (node instanceof DataNode) {
            DataNode dataNode = (DataNode) node;
            List<View> provides = dataNode.getProvides();
            if (provides != null) {
                for (View provide : provides) {
                    if ("urn:list-of-files".equals(provide.getUri())) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private Optional<String> getProperty(Node node, String uri) {
        if (node.getProperties() != null && node.getProperties() != null) {
            for (Property property : node.getProperties()) {
                if (uri.equals(property.getUri())) {
                    return Optional.of(property.getValue());
                }
            }
        }
        return Optional.empty();
    }

    private String getSize(Node node) {
        return getProperty(node, "ivo://ivoa.net/vospace/core#length")
                .map(value -> {
                    try {
                        long bytes = Long.parseLong(value);
                        return getHumanReadableSize(bytes);
                    } catch (NumberFormatException ex) {
                        LOG.warn("Invalid length for node " + node.getUri() + ". Length is " + value);
                        return "";
                    }
                })
                .orElse("");
    }

    /**
     * Credits: https://stackoverflow.com/a/16576773/771431
     */
    private String getHumanReadableSize(long bytes) {
        int u = 0;
        for (; bytes > 1024 * 1024; bytes >>= 10) {
            u++;
        }
        if (bytes > 1024) {
            u++;
        }
        return String.format("%.1f %cB", bytes / 1024f, " kMGTPE".charAt(u));
    }

    public boolean isFolder() {
        return "vos:ContainerNode".equals(type);
    }

    public boolean isFile() {
        return !"vos:ContainerNode".equals(type);
    }

    public String getPath() {
        return path;
    }

    public String getName() {
        return name;
    }

    public String getSize() {
        return size;
    }

    public String getCreator() {
        return creator;
    }

    public String getGroupRead() {
        return groupRead;
    }

    public String getGroupWrite() {
        return groupWrite;
    }

    public boolean isPublic() {
        return isPublic;
    }

    public boolean isAsyncTrans() {
        return asyncTrans;
    }

    public boolean isBusy() {
        return busy;
    }

    public boolean isListOfFiles() {
        return listOfFiles;
    }
}
