package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.ListNodeData;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import java.io.IOException;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import java.util.Arrays;
import java.util.List;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.Node;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

@Service
public class NodesService {

    private static final Logger LOG = LoggerFactory.getLogger(NodesService.class);

    @Autowired
    private VOSpaceClient client;

    @Value("${vospace-authority}")
    private String authority;

    public ListNodeData generateNodesHtml(String path, User user) {

        ListNodeData listNodeData = new ListNodeData();
        
        Node node = client.getNode(path);
        
        listNodeData.setWritable(NodeUtils.checkIfWritable(node, user.getName(), user.getGroups()));

        try (StringWriter sw = new StringWriter()) {

            if (node instanceof ContainerNode) {
                ContainerNode folder = (ContainerNode) node;
                sw.write("<tbody id=\"nodes\">");
                for (Node child : folder.getNodes()) {
                    sw.write(getNodeHtml(child, user));
                }
                sw.write("</tbody>");
            }

            listNodeData.setHtmlTable(sw.toString());
            
            return listNodeData;
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private String getNodeHtml(Node node, User user) {

        NodeInfo nodeInfo = new NodeInfo(node, authority);

        if (nodeInfo.isListOfFiles()) {
            // hidden file
            return "";
        }

        String html = "<tr>";
        html += "<td><input type=\"checkbox\" data-node=\"" + nodeInfo.getPath() + "\" ";
        if (nodeInfo.isAsyncTrans()) {
            html += "class=\"async\"";
        }
        html += "/></td>";
        html += "<td>" + getIcon(nodeInfo) + getLink(nodeInfo, user) + "</td>";
        html += "<td>" + nodeInfo.getSize() + "</td>";
        html += "<td>" + nodeInfo.getGroupRead() + "</td>";
        html += "<td>" + nodeInfo.getGroupWrite() + "</td>";
        html += "<td>";
        if (NodeUtils.checkIfWritable(node, user.getName(), user.getGroups())) {
            html += "<span class=\"icon trash-icon pointer\" onclick=\"deleteNode('" + nodeInfo.getPath() + "')\"></span>";
        }
        html += "</td>";
        html += "</tr>";
        return html;
    }

    private String getIcon(NodeInfo nodeInfo) {
        String html = "<span class=\"icon ";
        if (nodeInfo.isFile() && nodeInfo.isBusy()) {
            html += "gear";
        } else {
            if (nodeInfo.isFolder()) {
                html += "folder";
            } else {
                html += "file";
            }
            if (nodeInfo.isAsyncTrans()) {
                html += "-x";
            }
        }
        html += "-icon\"></span>&nbsp;";
        return html;
    }

    private String getLink(NodeInfo nodeInfo, User user) {
        if (isDownloadable(nodeInfo, user)) {
            if (nodeInfo.isFolder()) {
                return "<a href=\"#/nodes" + nodeInfo.getPath() + "\">" + nodeInfo.getName() + "</a>";
            } else {
                return "<a href=\"download" + nodeInfo.getPath() + "\" target=\"blank_\">" + nodeInfo.getName() + "</a>";
            }
        }
        return nodeInfo.getName();
    }

    private boolean isDownloadable(NodeInfo nodeInfo, User user) {
        if (nodeInfo.isFile()) {
            if (nodeInfo.isAsyncTrans() || nodeInfo.isBusy()) {
                return false;
            }
        }
        if (nodeInfo.isPublic()) {
            return true;
        }
        
        if (nodeInfo.getCreator().equals(user.getName())) {
            return true;
        }

        if (user.getGroups() != null && !user.getGroups().isEmpty() && !nodeInfo.getGroupRead().isEmpty()) {
            List<String> groupRead = Arrays.asList(nodeInfo.getGroupRead().split(" "));
            for (String group : groupRead) {
                if (user.getGroups().contains(group)) {
                    return true;
                }
            }
        }

        return false;
    }
}
