package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.Job;
import it.inaf.ia2.vospace.ui.exception.BadRequestException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.StructuredDataNode;
import net.ivoa.xml.vospace.v2.Transfer;
import net.ivoa.xml.vospace.v2.View;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.RestTemplate;

@RestController
public class JobController extends BaseController {

    @Value("${vospace-authority}")
    private String authority;

    @Autowired
    private VOSpaceClient client;

    @Autowired
    private RestTemplate restTemplate;

    @PostMapping(value = "/recall", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Job> startRecallFromTapeJob(@RequestBody List<String> paths) {

        if (paths.isEmpty()) {
            throw new BadRequestException("Received empty list of nodes");
        }

        String target;
        if (paths.size() == 1) {
            target = "vos://" + authority + paths.get(0);
        } else {
            target = createTempListOfFilesNode(paths);
        }

        Transfer transfer = new Transfer();
        transfer.setDirection("pullToVoSpace");
        transfer.setTarget(target);
        Protocol protocol = new Protocol();
        protocol.setUri("ia2:tape-recall");
        transfer.getProtocols().add(protocol);

        JobSummary job = client.startTransferJob(transfer);

        if (job.getPhase() == ExecutionPhase.QUEUED) {
            return ResponseEntity.ok(new Job(job));
        }
        // TODO: proper handling
        throw new RuntimeException("Error while executing job " + job.getJobId() + ". Job phase is " + job.getPhase() + ". QUEUED expected");
    }

    private String createTempListOfFilesNode(List<String> paths) {

        StructuredDataNode dataNode = createStructuredDataNode(paths);
        client.createNode(dataNode);
        String uploadEndpoint = getTempFileEndpoint(dataNode.getUri());

        String content = String.join("\n", paths);

        upload(uploadEndpoint, content);

        return dataNode.getUri();
    }

    private StructuredDataNode createStructuredDataNode(List<String> paths) {

        List<View> views = new ArrayList<>();
        View view = new View();
        view.setUri("urn:list-of-files");
        views.add(view);

        StructuredDataNode dataNode = new StructuredDataNode();

        String parentPath = getParentPath(paths);
        String newTempFile = ".tmp-" + UUID.randomUUID().toString().replace("-", "") + ".txt";

        dataNode.setUri("vos://" + authority + parentPath + "/" + newTempFile);

        dataNode.setAccepts(views);
        dataNode.setProvides(views);

        return dataNode;
    }

    private String getParentPath(List<String> paths) {
        // All the paths have the same parent, we can choose the first for extracting the path
        String firstPath = paths.get(0);
        return firstPath.substring(0, firstPath.lastIndexOf("/"));
    }

    private String getTempFileEndpoint(String target) {

        Transfer transfer = new Transfer();
        transfer.setDirection("pushToVoSpace");
        transfer.setTarget(target);
        Protocol protocol = new Protocol();
        protocol.setUri("ivo://ivoa.net/vospace/core#httpget");
        transfer.getProtocols().add(protocol);

        return client.getFileServiceEndpoints(transfer).get(0).getEndpoint();
    }

    private void upload(String endpoint, String content) {

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.MULTIPART_FORM_DATA);

        MultiValueMap<String, Object> parts = new LinkedMultiValueMap<>();
        parts.add("file", new MultipartFileResource(content, "list.txt"));

        HttpEntity<MultiValueMap<String, Object>> requestEntity = new HttpEntity<>(parts, headers);

        restTemplate.exchange(endpoint, HttpMethod.PUT, requestEntity, Void.class);
    }

    private class MultipartFileResource extends ByteArrayResource {

        private final String fileName;

        public MultipartFileResource(String content, String fileName) {
            super(content.getBytes());
            this.fileName = fileName;
        }

        @Override
        public String getFilename() {
            return this.fileName;
        }
    }

    @GetMapping(value = "/jobs", produces = MediaType.APPLICATION_JSON_VALUE)
    public List<Job> getJobs() {
        // TODO
        return new ArrayList<>();
    }
}
