package it.inaf.ia2.vospace.ui.controller;

import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.UploadFilesData;
import it.inaf.ia2.vospace.ui.exception.PermissionDeniedException;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import javax.validation.Valid;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class UploadController extends BaseController {

    @Value("${vospace-authority}")
    private String authority;

    @Autowired
    private VOSpaceClient client;

    @PostMapping(value = "/preupload", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<List<String>> prepareForUpload(@RequestBody @Valid UploadFilesData data) {

        if (getUser() == null) {
            throw new PermissionDeniedException("File upload not allowed to anonymous users");
        }

        CompletableFuture<String>[] calls
                = data.getFiles().stream().map(fileName -> prepareForDownload(getParentPath(data), fileName))
                        .toArray(CompletableFuture[]::new);

        List<String> uploadUrls = CompletableFuture.allOf(calls)
                .thenApplyAsync(ignore -> {
                    return Arrays.stream(calls).map(c -> c.join()).collect(Collectors.toList());
                }).join();

        return ResponseEntity.ok(uploadUrls);
    }

    private String getParentPath(UploadFilesData data) {
        String parentPath = data.getParentPath();
        if (!parentPath.startsWith("/")) {
            parentPath = "/" + parentPath;
        }
        return parentPath;
    }

    public CompletableFuture<String> prepareForDownload(String parentPath, String fileName) {

        return CompletableFuture.supplyAsync(() -> {
            String nodeUri = "vos://" + authority + parentPath + "/" + fileName;

            createDataNode(nodeUri, getUser().getName());

            return obtainUploadUrl(nodeUri);
        }, Runnable::run); // Passing current thread Executor to CompletableFuture to avoid "No thread-bound request found" exception
    }

    private void createDataNode(String nodeUri, String userId) {

        DataNode node = new DataNode();
        node.setUri(nodeUri);

        Property creator = new Property();
        creator.setUri("ivo://ivoa.net/vospace/core#creator");
        creator.setValue(userId);

        node.getProperties().add(creator);

        client.createNode(node);
    }

    private String obtainUploadUrl(String uri) {

        Transfer transfer = new Transfer();
        transfer.setDirection("pushToVoSpace");
        transfer.setTarget(uri);

        Protocol protocol = new Protocol();
        protocol.setUri("ivo://ivoa.net/vospace/core#httpget");
        transfer.getProtocols().add(protocol);

        return client.getFileServiceEndpoints(transfer).get(0).getEndpoint();
    }
}
