package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.vospace.ui.exception.VOSpaceException;
import java.util.Optional;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeInfo {

    private static final Logger LOG = LoggerFactory.getLogger(NodeInfo.class);

    private final String authority;

    private final String path;
    private final String name;
    private final String size;
    private final String type;
    private final String groupRead;
    private final String groupWrite;
    private final boolean isPublic;
    private final boolean asyncTrans;

    public NodeInfo(Node node, String authority) {
        this.authority = authority;
        this.path = getPath(node);
        this.name = path.substring(path.lastIndexOf("/") + 1);
        this.size = getSize(node);
        this.type = node.getType();
        this.groupRead = getGroupRead(node);
        this.groupWrite = getGroupWrite(node);
        this.isPublic = isPublic(node);
        this.asyncTrans = isAsyncTrans(node);
    }

    private String getPath(Node node) {

        String uri = node.getUri();

        String prefix = "vos://" + authority;

        if (!uri.startsWith(prefix)) {
            throw new VOSpaceException("Node authority is different from configured one! Configured is " + authority + ", but node URI is " + uri);
        }

        return uri.substring(prefix.length());
    }

    private String getGroupRead(Node node) {
        return getProperty(node, "ivo://ivoa.net/vospace/core#groupread").orElse("");
    }

    private String getGroupWrite(Node node) {
        return getProperty(node, "ivo://ivoa.net/vospace/core#groupwrite").orElse("");
    }

    private boolean isPublic(Node node) {
        return getProperty(node, "ivo://ivoa.net/vospace/core#ispublic").map(value -> "t".equals(value)).orElse(false);
    }

    private boolean isAsyncTrans(Node node) {
        return getProperty(node, "urn:async_trans").map(value -> "t".equals(value)).orElse(false);
    }

    private Optional<String> getProperty(Node node, String uri) {
        if (node.getProperties() != null && node.getProperties() != null) {
            for (Property property : node.getProperties()) {
                if (uri.equals(property.getUri())) {
                    return Optional.of(property.getValue());
                }
            }
        }
        return Optional.empty();
    }

    private String getSize(Node node) {
        return getProperty(node, "ivo://ivoa.net/vospace/core#length")
                .map(value -> {
                    try {
                        long bytes = Long.parseLong(value);
                        return getHumanReadableSize(bytes);
                    } catch (NumberFormatException ex) {
                        LOG.warn("Invalid length for node " + node.getUri() + ". Length is " + value);
                        return "";
                    }
                })
                .orElse("");
    }

    /**
     * Credits: https://stackoverflow.com/a/16576773/771431
     */
    private String getHumanReadableSize(long bytes) {
        int u = 0;
        for (; bytes > 1024 * 1024; bytes >>= 10) {
            u++;
        }
        if (bytes > 1024) {
            u++;
        }
        return String.format("%.1f %cB", bytes / 1024f, " kMGTPE".charAt(u));
    }

    public boolean isFolder() {
        return "vos:ContainerNode".equals(type);
    }

    public boolean isFile() {
        return !"vos:ContainerNode".equals(type);
    }

    public String getPath() {
        return path;
    }

    public String getName() {
        return name;
    }

    public String getSize() {
        return size;
    }

    public String getGroupRead() {
        return groupRead;
    }

    public String getGroupWrite() {
        return groupWrite;
    }

    public boolean isPublic() {
        return isPublic;
    }

    public boolean isAsyncTrans() {
        return asyncTrans;
    }
}
