package it.inaf.ia2.vospace.ui.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.service.NodesService;
import java.util.Arrays;
import java.util.List;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.Mockito;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.web.util.NestedServletException;

@SpringBootTest
@AutoConfigureMockMvc
public class NodesControllerTest {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    @MockBean
    private NodesService nodesService;

    @MockBean
    private VOSpaceClient client;

    @Autowired
    private MockMvc mockMvc;

    @Test
    public void testListNodesEmpty() throws Exception {

        mockMvc.perform(get("/nodes"))
                .andExpect(status().isOk());

        verify(nodesService).generateNodesHtml(eq("/"), any());
    }

    @Test
    public void testListNodesRoot() throws Exception {

        mockMvc.perform(get("/nodes/"))
                .andExpect(status().isOk());

        verify(nodesService).generateNodesHtml(eq("/"), any());
    }

    @Test
    public void testListNodesComplexPath() throws Exception {

        mockMvc.perform(get("/nodes/a/b/c"))
                .andExpect(status().isOk());

        verify(nodesService).generateNodesHtml(eq("/a/b/c"), any());
    }

    @Test
    public void testDeleteMultipleNodes() throws Exception {

        List<String> paths = Arrays.asList("/a/b/c", "/e/f/g");

        mockMvc.perform(post("/delete")
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(paths)))
                .andExpect(status().isNoContent());

        verify(client, times(1)).deleteNode(eq("/a/b/c"));
        verify(client, times(1)).deleteNode(eq("/e/f/g"));
    }

    @Test
    public void testErrorOnDelete() throws Exception {

        doThrow(new RuntimeException())
                .when(client).deleteNode(any());

        boolean exception = false;
        try {
            mockMvc.perform(post("/delete")
                    .contentType(MediaType.APPLICATION_JSON)
                    .content(MAPPER.writeValueAsString(Arrays.asList("/test"))))
                    .andReturn();
        } catch (NestedServletException ex) {
            exception = true;
        }

        assertTrue(exception);
    }
}
