/* Vuex store, for centralized state management */

import Vue from 'vue';
import Vuex from 'vuex';
import client from 'api-client';
import main from './main';

Vue.use(Vuex);

function updateArray(oldArr, newArr) {
  // empty the array
  oldArr.splice(0, oldArr.length);
  // fill again
  for (let i = 0; i < newArr.length; i++) {
    oldArr.push(newArr[i]);
  }
}

export default new Vuex.Store({
  state: {
    path: '',
    loading: true,
    asyncButtonEnabled: false,
    deleteButtonEnabled: false,
    jobs: [],
    jobsLoading: true,
    user: 'anonymous',
    nodesToDelete: [],
    writable: false
  },
  mutations: {
    setLoading(state, loading) {
      state.loading = loading;
    },
    setPath(state, value) {
      if (!value) {
        value = '';
      }
      state.path = value;
    },
    setAsyncButtonEnabled(state, value) {
      state.asyncButtonEnabled = value;
    },
    setDeleteButtonEnabled(state, value) {
      state.deleteButtonEnabled = value;
    },
    setJobs(state, jobs) {
      updateArray(state.jobs, jobs);
    },
    addJob(state, job) {
      state.jobs.push(job);
    },
    setJobsLoading(state, loading) {
      state.jobsLoading = loading;
    },
    setUsername(state, username) {
      state.user = username;
    },
    setNodesToDelete(state, paths) {
      updateArray(state.nodesToDelete, paths);
    },
    setWritable(state, value) {
      state.writable = value;
    }
  },
  actions: {
    setPath({ state, commit, dispatch }, path) {
      commit('setPath', path);
      client.getNode(state.path)
        .then(res => {
          commit('setWritable', res.writable);
          document.getElementById('nodes').outerHTML = res.htmlTable;
          let checkboxes = document.querySelectorAll('#nodes input[type="checkbox"]');
          for (let i = 0; i < checkboxes.length; i++) {
            checkboxes[i].addEventListener('change', function() {
              dispatch('computeButtonsVisibility');
            });
          }
          dispatch('computeButtonsVisibility');
        });
    },
    computeButtonsVisibility({ commit }) {
      commit('setAsyncButtonEnabled', document.querySelectorAll('#nodes input.async:checked').length > 0);
      commit('setDeleteButtonEnabled', document.querySelectorAll('#nodes input.deletable:checked').length > 0);
    },
    startAsyncRecallJob({ commit }) {
      let asyncCheckboxes = document.querySelectorAll('#nodes input.async:checked');
      let paths = [];
      for (let i = 0; i < asyncCheckboxes.length; i++) {
        paths.push(asyncCheckboxes[i].getAttribute('data-node'));
      }
      client.startAsyncRecallJob(paths)
        .then(job => {
          main.showInfo('Job started');
          commit('addJob', job);
        });
    },
    loadJobs({ commit }) {
      commit('setJobsLoading', true);
      client.loadJobs()
        .then(jobs => commit('setJobs', jobs))
        .finally(() => commit('setJobsLoading', false));
    },
    loadUserInfo({ commit }) {
      client.getUserInfo()
        .then(res => commit('setUsername', res.username));
    },
    createFolder({ state, dispatch }, newFolderName) {
      client.createFolder(state.path, newFolderName)
        .then(() => {
          // Reload current node
          dispatch('setPath', state.path);
        });
    },
    uploadFiles({ state, dispatch }, files) {
      let names = [];
      for (let file of files) {
        names.push(file.name);
      }
      client.prepareForUpload(state.path, names)
        .then(uploadUrls => {
          for (let i = 0; i < files.length; i++) {
            client.uploadFile(uploadUrls[i], files[i]);
          }
        })
        .then(() => {
          // Reload current node
          dispatch('setPath', state.path);
        });
    },
    deleteNodes({ state, dispatch }) {
      client.deleteNodes(state.nodesToDelete)
        .then(() => {
          // Reload current node
          dispatch('setPath', state.path);
        });
    }
  }
});
