/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.service;

import it.inaf.ia2.gms.client.GmsClient;
import it.inaf.ia2.gms.client.model.Permission;
import it.inaf.ia2.rap.client.ClientCredentialsRapClient;
import it.inaf.ia2.rap.data.Identity;
import it.inaf.ia2.rap.data.IdentityType;
import it.inaf.ia2.rap.data.RapUser;
import it.inaf.ia2.rap.data.TokenContext;
import it.inaf.ia2.vospace.ui.TokenProvider;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.ShareRequest;
import it.inaf.ia2.vospace.ui.data.SharingInfo;
import it.inaf.ia2.vospace.ui.exception.BadRequestException;
import it.inaf.ia2.vospace.ui.exception.VOSpaceStatusException;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import static it.inaf.oats.vospace.datamodel.NodeUtils.urlEncodePath;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

@Service
public class SharingService {

    private static final Logger LOG = LoggerFactory.getLogger(SharingService.class);

    private static final String SHARED_FILES_DIR_NAME = "Shared Files";

    @Value("${trusted.eppn.scope}")
    protected String trustedEppnScope;

    @Value("${vospace-authority}")
    protected String authority;

    private final Set<String> existingGroups = new CopyOnWriteArraySet<>();
    private final Set<String> existingPeopleGroups = new CopyOnWriteArraySet<>();
    private final Map<String, String> existingUsers = new ConcurrentHashMap<>();

    private final GmsClient gmsClient;
    private final ClientCredentialsRapClient rapClient;
    private final VOSpaceClient vospaceClient;
    private final TokenProvider tokenProvider;

    private TokenContext tokenContext;
    private Date lastUpdate;

    @Autowired
    public SharingService(GmsClient gmsClient, ClientCredentialsRapClient gmsRapClient,
            VOSpaceClient vospaceClient, TokenProvider tokenProvider) {
        this.gmsClient = gmsClient;
        this.rapClient = gmsRapClient;
        this.vospaceClient = vospaceClient;
        this.tokenProvider = tokenProvider;
    }

    public SharingInfo getSharingInfo() {

        loadInfoFromServices();

        SharingInfo sharingInfo = new SharingInfo();

        Set<String> people = new HashSet<>(existingUsers.values());
        people.addAll(existingPeopleGroups);

        sharingInfo.setPeople(new ArrayList<>(people));
        Collections.sort(sharingInfo.getPeople());

        sharingInfo.setGroups(new ArrayList<>(existingGroups));
        Collections.sort(sharingInfo.getGroups());

        return sharingInfo;
    }

    public void setNodeGroups(ShareRequest shareRequest) {

        loadInfoFromServices();

        createPeopleGroupIfNeeded(shareRequest.getUserRead());
        createPeopleGroupIfNeeded(shareRequest.getUserWrite());
        validateGroups(shareRequest.getGroupRead());
        validateGroups(shareRequest.getGroupWrite());

        String groupRead = getGroupString(shareRequest.getGroupRead(), shareRequest.getUserRead());
        String groupWrite = getGroupString(shareRequest.getGroupWrite(), shareRequest.getUserWrite());

        Optional<String> token = tokenProvider.getToken();

        Node node = vospaceClient.getNode(shareRequest.getPath(), token);
        getNodeProperty(node, NodeProperties.GROUP_READ_URI).setValue(groupRead);
        getNodeProperty(node, NodeProperties.GROUP_WRITE_URI).setValue(groupWrite);

        vospaceClient.setNode(node, shareRequest.isRecursive(), token);

        if (shareRequest.getNewPeople() != null && !shareRequest.getNewPeople().isEmpty()) {
            createSharedLinks(shareRequest);
        }
    }

    private void createSharedLinks(ShareRequest shareRequest) {

        for (String person : shareRequest.getNewPeople()) {

            loadAccessToken();

            if (nodeExists("/" + person)) {

                createSharedDir(person);

                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH.mm.ss");
                String linkNodeName = NodeUtils.getNodeName(shareRequest.getPath()) + " (" + sdf.format(new Date()) + ")";

                LinkNode sharedLink = new LinkNode();
                sharedLink.setUri("vos://" + authority + urlEncodePath("/" + person + "/" + SHARED_FILES_DIR_NAME + "/" + linkNodeName));

                try {
                    URI taregtUri = new URI("vos", authority, shareRequest.getPath(), null, null);
                    sharedLink.setTarget(taregtUri.toASCIIString());
                } catch (URISyntaxException ex) {
                    throw new RuntimeException(ex);
                }

                addPersonGroups(sharedLink, person);

                vospaceClient.createNode(sharedLink, Optional.of(tokenContext.getAccessToken()));
            }
        }
    }

    private boolean nodeExists(String path) {
        try {
            vospaceClient.getNode(path, Optional.of(tokenContext.getAccessToken()));
            return true;
        } catch (VOSpaceStatusException ex) {
            if (ex.getHttpStatus() == 404) {
                return false;
            } else {
                throw ex;
            }
        }
    }

    private void createSharedDir(String person) {
        ContainerNode sharedFilesDir = new ContainerNode();
        sharedFilesDir.setUri("vos://" + authority + urlEncodePath("/" + person + "/" + SHARED_FILES_DIR_NAME));
        addPersonGroups(sharedFilesDir, person);

        try {
            vospaceClient.createNode(sharedFilesDir, Optional.of(tokenContext.getAccessToken()));
        } catch (VOSpaceStatusException ex) {
            if (ex.getHttpStatus() != 200 && ex.getHttpStatus() != 409) { // created or already existing
                throw ex;
            }
        }
    }

    private void addPersonGroups(Node node, String person) {

        Property groupReadProp = new Property();
        groupReadProp.setUri(NodeProperties.GROUP_READ_URI);
        groupReadProp.setValue("people." + person.replace(".", "\\."));
        node.getProperties().add(groupReadProp);

        Property groupWriteProp = new Property();
        groupWriteProp.setUri(NodeProperties.GROUP_WRITE_URI);
        groupWriteProp.setValue("people." + person.replace(".", "\\."));
        node.getProperties().add(groupWriteProp);
    }

    private Property getNodeProperty(Node node, String uri) {
        for (Property property : node.getProperties()) {
            if (uri.equals(property.getUri())) {
                return property;
            }
        }
        Property property = new Property();
        property.setUri(uri);
        node.getProperties().add(property);
        return property;
    }

    private void createPeopleGroupIfNeeded(List<String> users) {
        for (String username : users) {
            if (!existingPeopleGroups.contains(username)) {

                String completeGroupName = "people." + username.replace(".", "\\.");
                String rapId = getRapId(username);

                gmsClient.createGroup(completeGroupName, true);
                gmsClient.addMember(completeGroupName, rapId);
                gmsClient.addPermission(completeGroupName, rapId, Permission.VIEW_MEMBERS);

                existingPeopleGroups.add(username);
            }
        }
    }

    private String getRapId(String username) {
        for (Map.Entry<String, String> entry : existingUsers.entrySet()) {
            if (entry.getValue().equals(username)) {
                return entry.getKey();
            }
        }
        throw new BadRequestException("Unable to find an identifier for user " + username);
    }

    private void validateGroups(List<String> groups) {
        for (String group : groups) {
            if (!existingGroups.contains(group)) {
                throw new BadRequestException("Group " + group + " doesn't exist");
            }
        }
    }

    private String getGroupString(List<String> groups, List<String> users) {
        List<String> values = new ArrayList<>(groups);
        for (String user : users) {
            values.add("people." + user.replace(".", "\\."));
        }
        return String.join(" ", values);
    }

    private void loadInfoFromServices() {
        loadAccessToken();

        if (lastUpdate == null || new Date().getTime() - lastUpdate.getTime() > 3600 * 100) {
            LOG.trace("Loading existing users and groups from services");
            CompletableFuture.allOf(
                    CompletableFuture.runAsync(() -> loadGroups()),
                    CompletableFuture.runAsync(() -> loadUsers())
            ).join();
        }
    }

    public void loadAccessToken() {
        if (tokenContext == null) {
            tokenContext = new TokenContext();
        }
        if (tokenContext.isTokenExpired()) {
            tokenContext.setAccessTokenResponse(rapClient.getAccessTokenFromClientCredentials());
            gmsClient.setAccessToken(tokenContext.getAccessToken());
        }
    }

    private void loadGroups() {
        List<String> groups = new ArrayList<>();
        List<String> peopleGroups = new ArrayList<>();

        for (String group : gmsClient.listGroups("", true)) {
            if (group.startsWith("people.")) {
                String singleUserGroup = group.substring("people.".length()).replace("\\.", ".");
                peopleGroups.add(singleUserGroup);
            } else {
                groups.add(group);
            }
        }

        existingGroups.addAll(groups);
        existingPeopleGroups.addAll(peopleGroups);
    }

    private void loadUsers() {
        for (RapUser user : rapClient.getUsers(trustedEppnScope, tokenContext)) {
            for (Identity identity : user.getIdentities()) {
                if (identity.getType() == IdentityType.EDU_GAIN
                        && identity.getEppn().endsWith("@" + trustedEppnScope)) {
                    String username = identity.getEppn().substring(0, identity.getEppn().indexOf("@"));
                    existingUsers.put(user.getId(), username.toLowerCase());
                    break;
                }
            }
        }
    }
}
