/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.vospace.ui.TokenProvider;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.MoveOrCopyRequest;
import it.inaf.ia2.vospace.ui.exception.VOSpaceException;
import java.util.Arrays;
import java.util.List;
import javax.servlet.http.HttpSession;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.util.ReflectionTestUtils;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.ResultActions;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.web.util.NestedServletException;

@SpringBootTest
@AutoConfigureMockMvc
@TestPropertySource(properties = {"vospace-authority=example.com!vospace", "pollingTimeout=3"})
public class NodesControllerTest {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    @MockBean
    private VOSpaceClient client;

    @MockBean
    private TokenProvider tokenProvider;

    @Autowired
    private NodesController nodesController;

    @Autowired
    private MockMvc mockMvc;

    @Test
    public void testListNodesEmpty() throws Exception {

        when(client.getNode(any(), any())).thenReturn(new DataNode());

        mockMvc.perform(get("/nodes"))
                .andExpect(status().isOk());

        verify(client, times(1)).getNode(eq("/"), any());
    }

    @Test
    public void testListNodesRoot() throws Exception {

        when(client.getNode(any(), any())).thenReturn(new DataNode());

        mockMvc.perform(get("/nodes/"))
                .andExpect(status().isOk());

        verify(client, times(1)).getNode(eq("/"), any());
    }

    @Test
    public void testListNodesComplexPath() throws Exception {

        when(client.getNode(any(), any())).thenReturn(new DataNode());

        mockMvc.perform(get("/nodes/a/b/c"))
                .andExpect(status().isOk());

        verify(client, times(1)).getNode(eq("/a/b/c"), any());
    }

    @Test
    public void testDeleteMultipleNodes() throws Exception {

        List<String> paths = Arrays.asList("/a/b/c", "/e/f/g");

        mockMvc.perform(post("/delete")
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(paths)))
                .andExpect(status().isNoContent());

        verify(client, times(1)).deleteNode(eq("/a/b/c"), any());
        verify(client, times(1)).deleteNode(eq("/e/f/g"), any());
    }

    @Test
    public void testErrorOnDelete() throws Exception {

        doThrow(new RuntimeException())
                .when(client).deleteNode(any(), any());

        boolean exception = false;
        try {
            mockMvc.perform(post("/delete")
                    .contentType(MediaType.APPLICATION_JSON)
                    .content(MAPPER.writeValueAsString(Arrays.asList("/test"))))
                    .andReturn();
        } catch (NestedServletException ex) {
            exception = true;
        }

        assertTrue(exception);
    }

    @Test
    public void testListNodesForMoveOrCopyModal() throws Exception {

        ContainerNode parent = new ContainerNode();
        parent.setUri("vos://example.com!vospace/a/b/c");

        ContainerNode child1 = new ContainerNode();
        child1.setUri("vos://example.com!vospace/a/b/c/c1");
        parent.getNodes().add(child1);

        DataNode child2 = new DataNode();
        child2.setUri("vos://example.com!vospace/a/b/c/c2");
        parent.getNodes().add(child2);

        ContainerNode child3 = new ContainerNode();
        child3.setUri("vos://example.com!vospace/a/b/c/c3");
        parent.getNodes().add(child3);

        ContainerNode child4 = new ContainerNode();
        child4.setUri("vos://example.com!vospace/a/b/c/c4");
        parent.getNodes().add(child4);

        LinkNode child5 = new LinkNode();
        child5.setUri("vos://example.com!vospace/a/b/c/c5");
        child5.setTarget("vos://example.com!vospace/a/b/c/c2");
        parent.getNodes().add(child5);

        LinkNode child6 = new LinkNode();
        child6.setUri("vos://example.com!vospace/a/b/c/c6");
        child6.setTarget("vos://example.com!vospace/a/b/c/c1");
        parent.getNodes().add(child6);

        LinkNode child7 = new LinkNode();
        child7.setUri("vos://example.com!vospace/a/b/c/c7");
        child7.setTarget("http://external-link");
        parent.getNodes().add(child7);

        when(client.getNode(eq("/a/b/c"), any())).thenReturn(parent);
        when(client.getNode(eq("/a/b/c/c1"), any())).thenReturn(child1);
        when(client.getNode(eq("/a/b/c/c2"), any())).thenReturn(child2);

        String response = mockMvc.perform(get("/nodesForMoveOrCopy")
                .param("path", "/a/b/c")
                .param("target", "/a/b/c/c3")
                .param("target", "/a/b/c/c4"))
                .andExpect(status().isOk())
                .andReturn().getResponse().getContentAsString();

        assertTrue(response.contains("c1")); // folder
        assertFalse(response.contains("c2")); // data node
        assertFalse(response.contains("c3")); // nodeToMove
        assertFalse(response.contains("c4")); // nodeToMove
        assertFalse(response.contains("c5")); // link to file
        assertTrue(response.contains("c6")); // link to container
        assertFalse(response.contains("c7")); // external link

        verify(client, times(1)).getNode(eq("/a/b/c"), any());
    }

    @Test
    public void testMoveNodeSuccess() throws Exception {

        when(client.getJobPhase(eq("job_id"), any()))
                .thenReturn(ExecutionPhase.EXECUTING)
                .thenReturn(ExecutionPhase.COMPLETED);

        testMoveNode()
                .andExpect(status().isOk())
                .andExpect(jsonPath("$[0].phase", is("COMPLETED")));
    }

    @Test
    public void testMoveNodeExecuting() throws Exception {

        when(client.getJobPhase(eq("job_id"), any()))
                .thenReturn(ExecutionPhase.EXECUTING);

        testMoveNode()
                .andExpect(status().isOk())
                .andExpect(jsonPath("$[0].phase", is("EXECUTING")));
    }

    @Test
    public void testMoveNodeError() throws Exception {

        when(client.getJobPhase(eq("job_id"), any()))
                .thenReturn(ExecutionPhase.ERROR);

        when(client.getErrorDetail(eq("job_id"), any())).thenReturn("move_error");

        try {
            testMoveNode();
            fail("Exception was expected");
        } catch (Exception ex) {
            assertTrue(ex.getCause() instanceof VOSpaceException);
            assertTrue(ex.getCause().getMessage().contains("move_error"));
        }
    }

    @Test
    public void testNewFolder() throws Exception {

        HttpSession mockedSession = mock(HttpSession.class);
        ReflectionTestUtils.setField(nodesController, "session", mockedSession);

        when(mockedSession.getAttribute("user_data")).thenReturn(mock(User.class));

        mockMvc.perform(post("/folder")
                .contentType(MediaType.APPLICATION_JSON)
                .content("{\"parentPath\": \"/parent\", \"name\": \"newFolder\"}"))
                .andExpect(status().is2xxSuccessful());

        verify(client, times(1)).createNode(any(), any());
    }

    @Test
    public void testListLinkNodes() throws Exception {

        ContainerNode parent = new ContainerNode();
        parent.setUri("vos://example.com!vospace");

        LinkNode link1 = new LinkNode();
        link1.setUri("vos://example.com!vospace/link1");
        link1.setTarget("vos://example.com!vospace/myfile");
        parent.getNodes().add(link1);

        LinkNode link2 = new LinkNode();
        link2.setUri("vos://example.com!vospace/link2");
        link2.setTarget("http://external-link");
        parent.getNodes().add(link2);

        DataNode linkedNode = new DataNode();
        linkedNode.setUri("vos://example.com!vospace/myfile");

        when(client.getNode(eq("/"), any())).thenReturn(parent);
        when(client.getNode(eq("/myfile"), any())).thenReturn(linkedNode);

        mockMvc.perform(get("/nodes"))
                .andExpect(status().isOk());

        verify(client, times(1)).getNode(eq("/"), any());
    }

    private ResultActions testMoveNode() throws Exception {

        JobSummary job = new JobSummary();
        job.setJobId("job_id");

        when(client.startTransferJob(any(), any())).thenReturn(job);

        MoveOrCopyRequest request = new MoveOrCopyRequest();
        request.setTargets(List.of("/path/to/target"));
        request.setDirection("/path/to/direction");

        return mockMvc.perform(post("/moveOrCopy")
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(request)));
    }
}
