/*
 * This file is part of vospace-ui
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.vospace.ui.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.vospace.ui.TokenProvider;
import it.inaf.ia2.vospace.ui.client.VOSpaceClient;
import it.inaf.ia2.vospace.ui.data.UploadFilesData;
import it.inaf.ia2.vospace.ui.exception.VOSpaceException;
import it.inaf.ia2.vospace.ui.exception.VOSpaceStatusException;
import java.util.List;
import java.util.Optional;
import static org.hamcrest.core.IsNull.nullValue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.Mock;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@AutoConfigureMockMvc
@TestPropertySource(properties = {"vospace-authority=example.com!vospace"})
public class UploadControllerTest {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    @MockBean
    private VOSpaceClient client;

    @MockBean
    private TokenProvider tokenProvider;

    @Autowired
    private MockMvc mockMvc;

    @Mock
    private User user;

    @BeforeEach
    public void setUp() {
        when(user.getName()).thenReturn("user_id");
        when(tokenProvider.getToken()).thenReturn(Optional.of("<token>"));
    }

    @Test
    public void testPreUpload() throws Exception {

        UploadFilesData data = new UploadFilesData();
        data.setParentPath("/mynode");
        data.setFiles(List.of("test.txt"));

        when(client.getFileServiceEndpoint(any(), any())).thenReturn("http://files/mynode/test.txt");

        mockMvc.perform(post("/preupload")
                .sessionAttr("user_data", user)
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(data)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$[0].url").value("http://files/mynode/test.txt"))
                .andExpect(jsonPath("$[0].error").value(nullValue()));
    }

    @Test
    public void testPreUploadRoot() throws Exception {

        UploadFilesData data = new UploadFilesData();
        data.setParentPath("/");
        data.setFiles(List.of("test.txt"));

        when(client.getFileServiceEndpoint(any(), any())).thenReturn("http://files/test.txt");

        mockMvc.perform(post("/preupload")
                .sessionAttr("user_data", user)
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(data)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$[0].url").value("http://files/test.txt"))
                .andExpect(jsonPath("$[0].error").value(nullValue()));
    }

    @Test
    public void testUploadNotAllowedToAnonymous() throws Exception {

        when(tokenProvider.getToken()).thenReturn(Optional.empty());

        UploadFilesData data = new UploadFilesData();
        data.setParentPath("/");
        data.setFiles(List.of("test.txt"));

        mockMvc.perform(post("/preupload")
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(data)))
                .andExpect(status().isForbidden());
    }

    @Test
    public void testDuplicatedNodeError() throws Exception {

        UploadFilesData data = new UploadFilesData();
        data.setParentPath("/mynode");
        data.setFiles(List.of("test.txt"));

        doThrow(new VOSpaceStatusException("Conflict", 409)).when(client).createNode(any(), any());

        mockMvc.perform(post("/preupload")
                .sessionAttr("user_data", user)
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(data)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$[0].error").value("Node already exists"))
                .andExpect(jsonPath("$[0].url").value(nullValue()));
    }

    @Test
    public void testGenericMetadataCreationError() throws Exception {

        UploadFilesData data = new UploadFilesData();
        data.setParentPath("/mynode");
        data.setFiles(List.of("test.txt"));

        doThrow(new VOSpaceStatusException("Server Error", 500)).when(client).createNode(any(), any());

        mockMvc.perform(post("/preupload")
                .sessionAttr("user_data", user)
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(data)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$[0].error").value("Unable to create node metadata"))
                .andExpect(jsonPath("$[0].url").value(nullValue()));
    }

    @Test
    public void testUploadUrlRecognizedError() throws Exception {

        UploadFilesData data = new UploadFilesData();
        data.setParentPath("/mynode");
        data.setFiles(List.of("test.txt"));

        doThrow(new VOSpaceException("Unable to connect")).when(client).getFileServiceEndpoint(any(), any());

        mockMvc.perform(post("/preupload")
                .sessionAttr("user_data", user)
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(data)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$[0].error").value("Unable to connect"))
                .andExpect(jsonPath("$[0].url").value(nullValue()));

        verify(client, times(1)).deleteNode(eq("/mynode/test.txt"), any());
    }

    @Test
    public void testUploadUrlFatalError() throws Exception {

        UploadFilesData data = new UploadFilesData();
        data.setParentPath("/mynode");
        data.setFiles(List.of("test.txt"));

        doThrow(new NullPointerException()).when(client).getFileServiceEndpoint(any(), any());

        mockMvc.perform(post("/preupload")
                .sessionAttr("user_data", user)
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(data)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$[0].error").value("Unable to obtain upload URL"))
                .andExpect(jsonPath("$[0].url").value(nullValue()));

        verify(client, times(1)).deleteNode(eq("/mynode/test.txt"), any());
    }

    @Test
    public void testUploadUrlErrorWithNodeDeletionError() throws Exception {

        UploadFilesData data = new UploadFilesData();
        data.setParentPath("/mynode");
        data.setFiles(List.of("test.txt"));

        doThrow(new NullPointerException()).when(client).getFileServiceEndpoint(any(), any());
        doThrow(new VOSpaceException("error")).when(client).deleteNode(any(), any());

        mockMvc.perform(post("/preupload")
                .sessionAttr("user_data", user)
                .contentType(MediaType.APPLICATION_JSON)
                .content(MAPPER.writeValueAsString(data)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$[0].error").value("Retrieval of upload URL failed. Manual cleanup of node metadata may be necessary"))
                .andExpect(jsonPath("$[0].url").value(nullValue()));

        verify(client, times(1)).deleteNode(eq("/mynode/test.txt"), any());
    }
}
