package it.inaf.ia2.vospace.ui.client;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.aa.data.User;
import it.inaf.ia2.vospace.ui.VOSpaceUiApplication;
import it.inaf.ia2.vospace.ui.data.Job;
import it.inaf.ia2.vospace.ui.exception.VOSpaceException;
import static it.inaf.oats.vospace.datamodel.NodeUtils.urlEncodePath;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import java.net.ConnectException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.util.List;
import java.util.Scanner;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ForkJoinPool;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.xml.bind.JAXB;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.uws.v1.Jobs;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Component
public class VOSpaceClient {

    private static final Logger LOG = LoggerFactory.getLogger(VOSpaceClient.class);

    @Value("${use-json}")
    private boolean useJson;

    @Value("${vospace-authority}")
    private String authority;

    private static final ObjectMapper MAPPER = new ObjectMapper();

    private final HttpClient httpClient;
    private final String baseUrl;
    private final ForkJoinPool jaxbExecutor;

    @Autowired
    protected HttpServletRequest servletRequest;

    public VOSpaceClient(@Value("${vospace-backend-url}") String backendUrl) {
        if (backendUrl.endsWith("/")) {
            // Remove final slash from configured URL
            backendUrl = backendUrl.substring(0, backendUrl.length() - 1);
        }
        baseUrl = backendUrl;

        jaxbExecutor = VOSpaceUiApplication.getJaxbExecutor();

        httpClient = HttpClient.newBuilder()
                .followRedirects(HttpClient.Redirect.ALWAYS)
                .version(HttpClient.Version.HTTP_1_1)
                .build();
    }

    public Node getNode(String path) {

        HttpRequest request = getRequest("/nodes" + urlEncodePath(path))
                .header("Accept", useJson ? "application/json" : "text/xml")
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> unmarshal(res, Node.class));
    }

    public JobSummary startTransferJob(Transfer transfer) {

        HttpRequest request = getRequest("/transfers?PHASE=RUN")
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .POST(HttpRequest.BodyPublishers.ofString(marshal(transfer)))
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> unmarshal(res, JobSummary.class));
    }

    public List<Protocol> getFileServiceEndpoints(Transfer transfer) {

        HttpRequest request = getRequest("/synctrans")
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .POST(HttpRequest.BodyPublishers.ofString(marshal(transfer)))
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> unmarshal(res, Transfer.class)).getProtocols();
    }

    public Node createNode(Node node) {

        String path = node.getUri().substring(("vos://" + authority).length());

        HttpRequest request = getRequest("/nodes" + urlEncodePath(path))
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .PUT(HttpRequest.BodyPublishers.ofString(marshal(node)))
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> unmarshal(res, Node.class));
    }

    public void deleteNode(String path) {

        HttpRequest request = getRequest("/nodes" + urlEncodePath(path))
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .DELETE()
                .build();

        call(request, BodyHandlers.ofInputStream(), 200, res -> null);
    }

    public List<Job> getJobs() {

        HttpRequest request = getRequest("/transfers?direction=pullToVoSpace")
                .header("Accept", useJson ? "application/json" : "text/xml")
                .header("Content-Type", useJson ? "application/json" : "text/xml")
                .GET()
                .build();

        return call(request, BodyHandlers.ofInputStream(), 200, res -> {
            return unmarshal(res, Jobs.class).getJobref().stream()
                    .map(jobDesc -> new Job(jobDesc))
                    .collect(Collectors.toList());
        });
    }

    private <T, U> U call(HttpRequest request, HttpResponse.BodyHandler<T> responseBodyHandler, int expectedStatusCode, Function<T, U> responseHandler) {
        try {
            return httpClient.sendAsync(request, responseBodyHandler)
                    .thenApply(response -> {
                        if (response.statusCode() == expectedStatusCode) {
                            return response.body();
                        }
                        logServerError(request, response);
                        throw new VOSpaceException("Error calling " + request.uri().toString() + ". Server response code is " + response.statusCode());
                    })
                    .thenApplyAsync(response -> responseHandler.apply(response), jaxbExecutor)
                    .join();
        } catch (CompletionException ex) {
            if (ex.getCause() != null) {
                if (ex.getCause() instanceof ConnectException) {
                    throw new VOSpaceException("Cannot connect to " + request.uri().getHost() + " on port " + request.uri().getPort());
                }
                if (ex.getCause() instanceof VOSpaceException) {
                    throw (VOSpaceException) ex.getCause();
                }
                LOG.error("Error calling " + request.uri().toString(), ex.getCause());
                throw new VOSpaceException("Error calling " + request.uri().toString(), ex.getCause());
            }
            LOG.error("Error calling " + request.uri().toString(), ex);
            throw new VOSpaceException("Error calling " + request.uri().toString(), ex);
        }
    }

    private HttpRequest.Builder getRequest(String path) {
        HttpRequest.Builder builder = HttpRequest.newBuilder(URI.create(baseUrl + path));
        String token = getToken();
        if (token != null) {
            builder.setHeader("Authorization", "Bearer " + token);
        }
        return builder;
    }

    private String getToken() {
        HttpSession session = servletRequest.getSession(false);
        if (session != null) {
            User user = (User) session.getAttribute("user_data");
            if (user != null) {
                return user.getAccessToken();
            }
        }
        return null;
    }

    private <T> T unmarshal(InputStream in, Class<T> type) {
        try {
            if (useJson) {
                return MAPPER.readValue(in, type);
            } else {
                return JAXB.unmarshal(in, type);
            }
        } catch (IOException ex) {
            LOG.error("Invalid JSON for class {}", type.getCanonicalName());
            throw new UncheckedIOException(ex);
        }
    }

    private String marshal(Object data) {
        try {
            if (useJson) {
                return MAPPER.writeValueAsString(data);
            } else {
                try ( StringWriter sw = new StringWriter()) {
                    JAXB.marshal(data, sw);
                    return sw.toString();
                }
            }
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private static <T> void logServerError(HttpRequest request, HttpResponse<T> response) {
        if (response.body() instanceof String) {
            logServerErrorString(request, (HttpResponse<String>) response);
        } else if (response.body() instanceof InputStream) {
            logServerErrorInputStream(request, (HttpResponse<InputStream>) response);
        } else {
            throw new UnsupportedOperationException("Unable to log error for response body type " + response.body().getClass().getCanonicalName());
        }
    }

    private static void logServerErrorString(HttpRequest request, HttpResponse<String> response) {

        LOG.error("Error while reading " + request.uri()
                + "\nServer response status code is " + response.statusCode()
                + "\nServer response text is " + response.body());
    }

    private static void logServerErrorInputStream(HttpRequest request, HttpResponse<InputStream> response) {
        Scanner s = new Scanner(response.body()).useDelimiter("\\A");
        String responseBody = s.hasNext() ? s.next() : "";
        String error = "Error while reading " + request.uri()
                + "\nServer response status code is " + response.statusCode()
                + "\nServer response text is " + responseBody;
        LOG.error(error);
    }
}
