package it.inaf.ia2.vospace.ui.client;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.http.HttpClient;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.CompletableFuture;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.vospace.v2.ContainerNode;
import static org.junit.jupiter.api.Assertions.assertEquals;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import static org.mockito.ArgumentMatchers.any;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.test.util.ReflectionTestUtils;

@ExtendWith(MockitoExtension.class)
public class VOSpaceClientTest {

    private HttpClient mockedHttpClient;
    private VOSpaceClient voSpaceClient;

    @BeforeEach
    public void init() {
        mockedHttpClient = mock(HttpClient.class);

        HttpClient.Builder builder = mock(HttpClient.Builder.class);
        when(builder.followRedirects(any())).thenReturn(builder);
        when(builder.version(any())).thenReturn(builder);
        when(builder.build()).thenReturn(mockedHttpClient);

        try ( MockedStatic<HttpClient> staticMock = Mockito.mockStatic(HttpClient.class)) {
            staticMock.when(HttpClient::newBuilder).thenReturn(builder);
            voSpaceClient = new VOSpaceClient("http://localhost/vospace");
        }

        voSpaceClient.servletRequest = mock(HttpServletRequest.class);
    }

    @Test
    public void testGetXmlNode() {
        ReflectionTestUtils.setField(voSpaceClient, "useJson", false);

        CompletableFuture response = getMockedStreamResponseFuture(200, getResourceFileContent("nodes-response.xml"));
        when(mockedHttpClient.sendAsync(any(), any())).thenReturn(response);

        ContainerNode node = (ContainerNode) voSpaceClient.getNode("/node1");
        assertEquals("vos://ia2.inaf.it!vospace/node1", node.getUri());
    }

    protected static String getResourceFileContent(String fileName) {
        try ( InputStream in = VOSpaceClientTest.class.getClassLoader().getResourceAsStream(fileName)) {
            return new String(in.readAllBytes(), StandardCharsets.UTF_8);
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    protected static CompletableFuture<HttpResponse<InputStream>> getMockedStreamResponseFuture(int statusCode, String body) {
        return CompletableFuture.completedFuture(getMockedStreamResponse(200, body));
    }

    protected static HttpResponse<InputStream> getMockedStreamResponse(int statusCode, String body) {
        HttpResponse response = getMockedResponse(statusCode);
        InputStream in = new ByteArrayInputStream(body.getBytes(StandardCharsets.UTF_8));
        when(response.body()).thenReturn(in);
        return response;
    }

    protected static HttpResponse getMockedResponse(int statusCode) {
        HttpResponse response = mock(HttpResponse.class);
        when(response.statusCode()).thenReturn(statusCode);
        return response;
    }
}
